import jdatetime
import json
from rest_framework import serializers
from django.utils import timezone
from apps.appointments.models import DoctorSchedule,PatientAppointment
from datetime import datetime,timedelta
from apps.doctor.api.v1.serializers import DoctorPublicProfileSerializer
from apps.patient.api.v1.serializers import PatientProfileSerializer
from apps.appointments.utils import generate_batch_id, get_last_day_of_year

class RecordSerializer(serializers.Serializer):
    start_time = serializers.TimeField()
    end_time = serializers.TimeField()
    booked = serializers.BooleanField(default=False,read_only=True, required=False)

class DoctorScheduleSerializer(serializers.ModelSerializer):
    records = serializers.ListField(
        child=RecordSerializer()  # Use the nested serializer for each item in the list
    , required=False)
    repeat = serializers.CharField(required=False)
    date = serializers.CharField()

    class Meta:
        model = DoctorSchedule
        fields = ['date', 'repeat', 'records',]
    
    
    def validate(self, data):
        if not self.context['doctor'].visit_time:
            raise serializers.ValidationError("Please first set time_visit for doctor")
        original_date = data.get('date')
        try:
            jalali_date = jdatetime.date(year=int(original_date[:4]), month=int(original_date[4:6]), day=int(original_date[6:8]))
        except:
            raise serializers.ValidationError('date', 'incorrect format. expected format is YYYYMMDD.')
        data['date'] = jalali_date.togregorian()
        
        repeat_values = ['none', 'weeekly', 'daily', 'daily_except_friday', 'every_other_day', 'every_other_day_except_friday', 'every_other_week']
        if data.get('repeat', 'none') not in repeat_values:
            raise serializers.ValidationError('repeat', f'Valid options for repeat are {repeat_values}')
        records = data.get('records', [])
        if not records:
            raise serializers.ValidationError({"records": "This field is required"})
        for record in records:
            start_time = record.get('start_time')
            end_time = record.get('end_time')
            if start_time >= end_time:
                raise serializers.ValidationError("Start time must be before end time for each record.")
        records.sort(key=lambda x: x['start_time'])
        for i in range(len(records) - 1):
            current_record = records[i]
            next_record = records[i + 1]
            current_end_time = current_record['end_time']
            next_start_time = next_record['start_time']

            if current_end_time > next_start_time:
                raise serializers.ValidationError('امکان ثبت به دلیل تداخل زمانی وجود ندارد')
        return data
 
    def create(self, validated_data):
        records_data = validated_data.get('records')
        repeat_choice = validated_data.get('repeat')
        doctor = self.context['doctor'] 
        validated_data['doctor'] = doctor
        batch_id = generate_batch_id()
        all_time_slots = []
        for record in records_data:
            start_time = record.get('start_time')
            end_time = record.get('end_time')
            start_time = datetime.combine(validated_data.get('date'), start_time)
            end_time = datetime.combine(validated_data.get('date'), end_time)
            time_slots = []
            while start_time < end_time:
                next_time = start_time + timedelta(minutes=doctor.visit_time)
                time_slots.append({
                        "start_time": start_time.strftime('%H:%M:%S'), 
                        "end_time": next_time.strftime('%H:%M:%S'),
                         "booked":False
                    })
                start_time = next_time
            all_time_slots.append(time_slots)

            record['start_time'] = record['start_time'].strftime('%H:%M:%S')
            record['end_time'] = record['end_time'].strftime('%H:%M:%S')

        if repeat_choice=='none':
            appointments = []
            for i in range(len(validated_data['records'])):
                appointment = DoctorSchedule.objects.create(
                    doctor=self.context['doctor'],
                    start_time=validated_data['records'][i]['start_time'],
                    end_time=validated_data['records'][i]['end_time'],
                    date=validated_data['date'],
                    batch_id=batch_id,
                    time_slots=all_time_slots[i]
                )
                appointments.append(appointment)
            return appointments
        
        elif repeat_choice=='weekly':
            last_day_of_year = get_last_day_of_year(jdatetime.date.fromgregorian(date=validated_data['date']))
            last_day_of_year = last_day_of_year.togregorian()
            current_date=validated_data['date']
            appointments = []
            while current_date <= last_day_of_year:
                for i in range(len(validated_data['records'])):
                    appointment = DoctorSchedule.objects.create(
                        doctor=self.context['doctor'],
                        start_time=validated_data['records'][i]['start_time'],
                        end_time=validated_data['records'][i]['end_time'],
                        date=current_date,
                        batch_id=batch_id,
                        time_slots=all_time_slots[i]
                    )
                    appointments.append(appointment)

                current_date += timedelta(days=7)

            return appointments
        
        elif repeat_choice=='daily':
            last_day_of_year = get_last_day_of_year(jdatetime.date.fromgregorian(date=validated_data['date']))
            last_day_of_year = last_day_of_year.togregorian()
            current_date=validated_data['date']
            appointments = []
            while current_date <= last_day_of_year:
                for i in range(len(validated_data['records'])):
                    appointment = DoctorSchedule.objects.create(
                        doctor=self.context['doctor'],
                        start_time=validated_data['records'][i]['start_time'],
                        end_time=validated_data['records'][i]['end_time'],
                        date=current_date,
                        batch_id=batch_id,
                        time_slots=all_time_slots[i]
                    )
                    appointments.append(appointment)

                current_date += timedelta(days=1)  
            
            return appointments
        
        elif repeat_choice == 'daily_except_friday':
            last_day_of_year = get_last_day_of_year(jdatetime.date.fromgregorian(date=validated_data['date']))
            last_day_of_year = last_day_of_year.togregorian()
            current_date=validated_data['date']
            appointments = []
            while current_date <= last_day_of_year:
                if current_date.weekday() != 4:
                    for i in range(len(validated_data['records'])):
                        appointment = DoctorSchedule.objects.create(
                            doctor=self.context['doctor'],
                            start_time=validated_data['records'][i]['start_time'],
                            end_time=validated_data['records'][i]['end_time'],
                            date=current_date,
                            batch_id=batch_id,
                            time_slots=all_time_slots[i]
                        )
                        appointments.append(appointment)

                current_date += timedelta(days=1)

            return appointments
        elif repeat_choice == 'every_other_day':
            last_day_of_year = get_last_day_of_year(jdatetime.date.fromgregorian(date=validated_data['date']))
            last_day_of_year = last_day_of_year.togregorian()
            current_date=validated_data['date']
            appointments = []
            while current_date <= last_day_of_year:                
                for i in range(len(validated_data['records'])):
                    appointment = DoctorSchedule.objects.create(
                        doctor=self.context['doctor'],
                        start_time=validated_data['records'][i]['start_time'],
                        end_time=validated_data['records'][i]['end_time'],
                        date=current_date,
                        batch_id=batch_id,
                        time_slots=all_time_slots[i]
                    )
                    appointments.append(appointment)

                current_date += timedelta(days=2)  # Increment by 2 days for every other day

            return appointments
        elif repeat_choice == 'every_other_day_except_friday':
            last_day_of_year = get_last_day_of_year(jdatetime.date.fromgregorian(date=validated_data['date']))
            last_day_of_year = last_day_of_year.togregorian()
            current_date=validated_data['date']
            appointments = []
            while current_date <= last_day_of_year:
                if current_date.weekday() != 4:  # Skip Fridays (weekday 6 in JalaliDate)
                    for i in range(len(validated_data['records'])):
                        appointment = DoctorSchedule.objects.create(
                            doctor=self.context['doctor'],
                            start_time=validated_data['records'][i]['start_time'],
                            end_time=validated_data['records'][i]['end_time'],
                            date=current_date,
                            batch_id=batch_id,
                            time_slots=all_time_slots[i]
                        )
                        appointments.append(appointment)

                current_date += timedelta(days=2)  # Increment by 2 days for every other day
            return appointments
        elif repeat_choice == 'every_other_week':
            last_day_of_year = get_last_day_of_year(jdatetime.date.fromgregorian(date=validated_data['date']))
            last_day_of_year = last_day_of_year.togregorian()
            current_date=validated_data['date']
            appointments = []
            while current_date <= last_day_of_year:
                for i in range(len(validated_data['records'])):
                    appointment = DoctorSchedule.objects.create(
                        doctor=self.context['doctor'],
                        start_time=validated_data['records'][i]['start_time'],
                        end_time=validated_data['records'][i]['end_time'],
                        date=current_date,
                        batch_id=batch_id,
                        time_slots=all_time_slots[i]
                    )
                    appointments.append(appointment)
                current_date += timedelta(days=14)  # Increment by 14 days for every other week

            return appointments
        
        raise serializers.ValidationError("Invalid `repeat` value")

class ScheduleViewSerializer(serializers.ModelSerializer):
    class Meta:
        model = DoctorSchedule
        fields = ['date', 'time_slots', 'start_time', 'end_time' ]
    
    def to_representation(self, instance):
        data = super().to_representation(instance)
        data['date'] = jdatetime.date.fromgregorian(date=instance.date).strftime('%Y%m%d')
        return data

class PatientAppointmentSerializer(serializers.ModelSerializer):
    patient = PatientProfileSerializer(required=False)
    doctor = DoctorPublicProfileSerializer(required=False)
    datetime = serializers.DateTimeField(
        input_formats=['%Y%m%d-%H:%M:%S', '%Y%m%d-%H:%M'],
        format=None  # Let us handle formatting in to_representation
    )

    class Meta:
        model = PatientAppointment
        fields = ['id', 'patient_name', 'datetime', 'accepted', 'patient', 'doctor']
        extra_kwargs = {
            "patient_name": {"required": False},
            "patient": {"read_only": True},
            "doctor": {"read_only": True},
            "accepted": {"read_only": True}
        }
      
    def validate(self, data):
        doctor = self.context.get('doctor')
        appointment_datetime = data['datetime']
        date = appointment_datetime.date()
        time = appointment_datetime.time()
        jalali_date = jdatetime.date(year=date.year, month=date.month, day=date.day)
        gregorian_date = jalali_date.togregorian()
        doctor_appointment = DoctorSchedule.objects.filter(doctor=doctor, date=gregorian_date, start_time__lte=time, end_time__gt=time).first()
        if not doctor_appointment:
            raise serializers.ValidationError("وقت خالی در زمان درخواست شده وجود ندارد")
  
        time_str = time.strftime('%H:%M:%S')

        time_slot = None
        for slot in doctor_appointment.time_slots:
            if slot['start_time'] == time_str :
                time_slot = slot
                break
        if time_slot is None:
            raise serializers.ValidationError("زمان انتخاب شده در دسترس نیست")
        
        if time_slot['booked']:
            raise serializers.ValidationError("زمان انتخاب شده قبلا رزرو شده است")
        return data

    def create(self, validated_data):
        doctor = self.context['doctor']
        patient = self.context['patient']
        appointment_datetime = validated_data['datetime']
        date = appointment_datetime.date()
        time = appointment_datetime.time()
        jalali_date = jdatetime.date(year=date.year, month=date.month, day=date.day)
        gregorian_date = jalali_date.togregorian()
        # Update the doctor's time slot to booked
        # TODO: account for race conditions use get for update
        try:
            doctor_appointment = DoctorSchedule.objects.get(doctor=doctor, date=gregorian_date, start_time__lte=time, end_time__gt=time)
        except DoctorSchedule.DoesNotExist:
            raise serializers.ValidationError("وقت خالی در زمان درخواست شده وجود ندارد")
        time_str = time.strftime('%H:%M:%S')
        booked_slot = None
        for slot in doctor_appointment.time_slots:
            if slot['start_time'] <= time_str < slot['end_time'] and slot['booked'] == False:
                slot['booked'] = True
                booked_slot = slot
                break
        if not booked_slot:
            raise serializers.ValidationError("نوبت خالی در زمان درخواست شده وجود ندارد.")
        doctor_appointment.save()

        combined_datetime = datetime.combine(gregorian_date, datetime.strptime(booked_slot['start_time'], "%H:%M:%S").time())
        # Make sure to localize the datetime to the current timezone
        combined_datetime = timezone.make_aware(combined_datetime, timezone.get_current_timezone())
        appointment = PatientAppointment.objects.create(
            doctor=doctor,
            patient=patient,
            patient_name=validated_data.get('patient_name', patient.full_name),
            datetime = combined_datetime
        )
        return appointment

    def to_representation(self, instance):
        data = super().to_representation(instance)
        # Convert to local time before converting to Jalali
        local_dt = timezone.localtime(instance.datetime)
        data['datetime'] = jdatetime.datetime.fromgregorian(datetime=local_dt).strftime('%Y%m%d-%H:%M:%S')
        return data
