from rest_framework.views import APIView
from rest_framework import status, permissions
from rest_framework.response import Response
from django.shortcuts import get_object_or_404
from apps.doctor.permissions import IsDoctor
from apps.appointments.models import DoctorSchedule, PatientAppointment
from apps.patient.models import Patient
from apps.doctor.models import Doctor
from .serializers import DoctorScheduleSerializer, ScheduleViewSerializer, PatientAppointmentSerializer
from django.utils import timezone
import jdatetime
from datetime import datetime, timedelta

class ScheduleListCreateAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request):
        doctor = get_object_or_404(Doctor, user=request.user)
        schedule_records = DoctorSchedule.objects.filter(doctor=doctor, date__gte=timezone.now().date())
        serializer = ScheduleViewSerializer(schedule_records,many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)
    

    def post(self, request):
        doctor = get_object_or_404(Doctor, user=request.user)
        serializer = DoctorScheduleSerializer(data=request.data,context={'doctor': doctor,'request': request})
        if serializer.is_valid():
            records = serializer.save()
            return Response(ScheduleViewSerializer(records, many=True).data, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class DoctorScheduleAPIView(APIView):
    permission_classes = [permissions.AllowAny]

    def get(self, request, doctor_id):
        today = timezone.now()
        next_21_days = today + timedelta(days=21)
        
        appointments = DoctorSchedule.objects.filter(
            doctor=doctor_id,
            date__range=(today, next_21_days)
        ).order_by('date')
        serializer =ScheduleViewSerializer(appointments, many=True)
        
        return Response(serializer.data, status=status.HTTP_200_OK)

class BookAppointmentView(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def post(self, request, doctor_id):
        doctor = get_object_or_404(Doctor, id=doctor_id)
        patient=get_object_or_404(Patient, user=request.user)
        serializer = PatientAppointmentSerializer(data=request.data, context={'doctor': doctor,'patient':patient})

        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class PatientAppointmentsView(APIView):
    permission_classes = [permissions.IsAuthenticated]
    def get(self, request):
        future = request.query_params.get('future', None)
        accepted_only = request.query_params.get('accepted_only', None)
        appointments = PatientAppointment.objects.filter(patient__user=request.user)
        if future:
            appointments = appointments.filter(datetime__gte=timezone.now())
        if accepted_only:
            appointments = appointments.filter(accepted=True)
        serializer = PatientAppointmentSerializer(appointments, many=True)
        
        return Response(serializer.data, status=status.HTTP_200_OK)

class DoctorAppointmentsView(APIView):
    permission_classes = [IsDoctor]    
    def get(self, request):
        unseen_appointments = PatientAppointment.objects.filter(doctor__user=request.user, datetime__gte=timezone.now().date(), accepted__isnull=True)
        all_appointments = PatientAppointment.objects.filter(doctor__user=request.user)
        next_appointments = PatientAppointment.objects.filter(doctor__user=request.user, datetime__gte=timezone.now().date(), accepted=True)
        # serializer = PatientAppointmentSerializer(appointments, many=True)
        return Response({
            "all_appointments": PatientAppointmentSerializer(all_appointments, many=True).data,
            "next_appointments": PatientAppointmentSerializer(next_appointments, many=True).data,
            "unseen_appointments": PatientAppointmentSerializer(unseen_appointments, many=True).data
        }, status=status.HTTP_200_OK)

class AcceptAppointmentAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request, app_id):
        appointment = get_object_or_404(PatientAppointment, pk=app_id, doctor__user=request.user)
        if appointment.accepted == True or appointment.accepted == False:
            return Response({"error": "already set"}, status=status.HTTP_400_BAD_REQUEST)
        appointment.accepted = True
        appointment.save()
        return Response(PatientAppointmentSerializer(appointment).data, status=status.HTTP_200_OK)

class RejectAppointmentAPIView(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def get(self, request, app_id):
        appointment = get_object_or_404(PatientAppointment, pk=app_id, doctor__user=request.user)
        if appointment.accepted == True or appointment.accepted == False:
            return Response({"error": "already set"}, status=status.HTTP_400_BAD_REQUEST)
        if appointment.datetime < timezone.now():
            appointment.accepted = False
            appointment.save()
        else:
            local_dt = timezone.localtime(appointment.datetime)
            schedule_record = DoctorSchedule.objects.get(doctor__user=request.user, date=appointment.datetime.date(), start_time__lte=local_dt.time(), end_time__gt=local_dt.time())
            time_str = local_dt.time().strftime('%H:%M:%S')
            for slot in schedule_record.time_slots:
                if slot['start_time'] == time_str:
                    slot['booked'] = False
                    break
            schedule_record.save()
            appointment.accepted = False
            appointment.save()
        return Response(PatientAppointmentSerializer(appointment).data, status=status.HTTP_200_OK)
