from django.core.validators import RegexValidator
from django.utils.translation import gettext_lazy as _
from django.utils import timezone
from django.contrib.auth import get_user_model
from rest_framework.exceptions import PermissionDenied
from rest_framework import serializers
from apps.core.utils import send_sms
from apps.core.models import File
from apps.user.serializers import SignUp_LoginSerializer as UserAuthSerializer, OTPVerificationSerializer as UserOTPSerializer
from apps.patient.models import Patient
from apps.doctor.models import (Doctor, DoctorWorkRecord, DoctorEducationHistory,
                                DoctorCertificate, DoctorAward,)
from apps.doctor import utils

User = get_user_model()

class SignUp_LoginSerializer(UserAuthSerializer):
    USER_DETAIL_MODEL = Doctor

class OTPReferralVerificationSerializer(serializers.Serializer):
    mobile_phone = serializers.CharField(max_length=11)
    otp = serializers.CharField()
    referral_code = serializers.CharField()
    medical_system_code = serializers.CharField()

    def validate(self, values):
        values = super().validate(values)
        try:
            user = User.objects.get(mobile_phone=values['mobile_phone'], otp=values['otp'])
        except User.DoesNotExist:
            raise serializers.ValidationError("کد یکبار مصرف صحیح نیست.")
        
        referral_doctor = Doctor.objects.filter(identification_code=values['referral_code']).first()
        medical_system_code = values.get("medical_system_code")

        if Doctor.objects.filter(medical_system_code=medical_system_code).exists():
            Doctor.objects.filter(user=user).delete()
            user.delete()
            raise serializers.ValidationError("پزشکی با این کد نظام پزشکی قبلاً ثبت شده است.")

        if not user.is_first_time_login():
            raise serializers.ValidationError('امکان ارزیابی کد یکبار مصرف وجود ندارد')

        if user.is_otp_expired():
            # TODO: Check how many times user has requested an otp in the past 5 hours before sending a new otp

            new_otp = user.generate_otp()
            send_sms(user.mobile_phone, new_otp)
            raise serializers.ValidationError("کد یکبار مصرف منقضی شده است . کد دیگری برای شما ارسال خواهد شد")

        values["doctor"] = user.doctor
        return values

    def save(self, **kwargs):
        user = self.validated_data['doctor'].user
        user.last_login = timezone.now()
        user.save()
        doctor = self.validated_data['doctor']
        doctor_identification_code = utils.generate_unique_identification_code()
        doctor.identification_code = doctor_identification_code
        doctor.invited_by_code = self.validated_data['referral_code']
        doctor.medical_system_code = self.validated_data['medical_system_code']
        doctor.save()
        return doctor

class OTPVerificationSerializer(UserOTPSerializer):
    USER_TYPE = 'doctor'

class WorkRecordSerializer(serializers.ModelSerializer):
    image_id = serializers.ListField(
        child=serializers.IntegerField(), write_only=True)
    image_url = serializers.SerializerMethodField(read_only=True)
    image_ids = serializers.SerializerMethodField(read_only=True)

    class Meta:
        model = DoctorWorkRecord

        fields = ['id', 'doctor', 'title', 'dec', 'is_current', 'from_date', 'to_date', 'created_at', 'updated_at',
                  'image_id', 'image_url', 'image_ids']
        extra_kwargs = {
            'doctor': {'read_only': True},
        }

    def get_image_url(self, obj):
        return [image.file.url for image in obj.image.all()]

    def get_image_ids(self, obj):
        return [image.id for image in obj.image.all()]

    def create(self, validated_data):
        image_ids = validated_data.pop('image_id', [])
        images = File.objects.filter(id__in=image_ids)
        doctor = self.context.get('doctor')
        for image in images:
            if image.user != doctor.user:
                raise serializers.ValidationError(f"Invalid image id {image.id}")

        work_record = DoctorWorkRecord.objects.create(doctor=doctor, **validated_data)
        work_record.image.set(images)

        return work_record

    def update(self, instance, validated_data):

        image_ids = validated_data.pop('image_id', None)
        doctor = self.context.get('doctor')

        for attr, value in validated_data.items():
            setattr(instance, attr, value)

        if image_ids is not None:
            current_images = set({image.id for image in instance.image.all()})
            new_images = set(image_ids)

            images_to_remove = current_images - new_images

            if images_to_remove:
                images_to_remove = File.objects.filter(id__in=images_to_remove)
                instance.image.remove(*images_to_remove)
                for image in images_to_remove:
                    if not DoctorWorkRecord.objects.filter(image=image).exists():
                        image.delete()

            images_to_add = new_images - current_images

            if images_to_add:
                for image_id in images_to_add:
                    image = File.objects.get(id=image_id)
                    if image.user != doctor.user:
                        serializers.ValidationError(f"Invalid image id {image.id}")
                    instance.image.add(image)

        instance.save()
        return instance

class EducationHistorySerializer(serializers.ModelSerializer):
    image_id = serializers.ListField(
        child=serializers.IntegerField(), write_only=True)
    image_url = serializers.SerializerMethodField(read_only=True)
    image_ids = serializers.SerializerMethodField(read_only=True)

    class Meta:
        model = DoctorEducationHistory
        fields = ['id', 'doctor', 'title', 'dec', 'from_date', 'to_date', 'created_at', 'updated_at', 'image_id',
                  'image_url', 'image_ids']
        extra_kwargs = {
            'doctor': {'read_only': True}
        }

    def get_image_url(self, obj):
        return [image.file.url for image in obj.image.all()]

    def get_image_ids(self, obj):
        return [image.id for image in obj.image.all()]

    def create(self, validated_data):
        image_ids = validated_data.pop('image_id', [])
        images = File.objects.filter(id__in=image_ids)
        doctor = self.context.get('doctor')

        for image in images:
            if image.user != doctor.user:
                raise serializers.ValidationError(f"Invalid image id {image.id}")

        education_history = DoctorEducationHistory.objects.create(doctor=doctor, **validated_data)
        education_history.image.set(images)

        return education_history

    def update(self, instance, validated_data):

        image_ids = validated_data.pop('image_id', None)
        doctor = self.context.get('doctor')

        for attr, value in validated_data.items():
            setattr(instance, attr, value)

        if image_ids is not None:
            current_images = set({image.id for image in instance.image.all()})
            new_images = set(image_ids)

            images_to_remove = current_images - new_images

            if images_to_remove:
                images_to_remove = File.objects.filter(id__in=images_to_remove)
                instance.image.remove(*images_to_remove)
                for image in images_to_remove:
                    if not DoctorEducationHistory.objects.filter(image=image).exists():
                        image.delete()

            images_to_add = new_images - current_images

            if images_to_add:
                for image_id in images_to_add:
                    image = File.objects.get(id=image_id)
                    if image.user != doctor.user:
                        raise serializers.ValidationError(f"Invalid image id {image.id}")
                    instance.image.add(image)

        instance.save()
        return instance


class DoctorCertificateSerializer(serializers.ModelSerializer):
    image_id = serializers.ListField(child=serializers.IntegerField(), write_only=True)
    image_url = serializers.SerializerMethodField(read_only=True)
    image_ids = serializers.SerializerMethodField(read_only=True)

    class Meta:
        model = DoctorCertificate
        fields = ['id', 'doctor', 'title', 'dec', 'created_at', 'updated_at', 'image_id', 'image_url', 'image_ids']
        extra_kwargs = {
            'doctor': {'read_only': True}
        }

    def get_image_url(self, obj):
        return [image.file.url for image in obj.image.all()]

    def get_image_ids(self, obj):
        return [image.id for image in obj.image.all()]

    def create(self, validated_data):
        image_ids = validated_data.pop('image_id', [])
        images = File.objects.filter(id__in=image_ids)
        doctor = self.context.get('doctor')

        for image in images:
            if image.user != doctor.user:
                raise serializers.ValidationError(f"Invalid image id {image.id}")

        certificate = DoctorCertificate.objects.create(doctor=doctor, **validated_data)
        certificate.image.set(images)

        return certificate

    def update(self, instance, validated_data):

        image_ids = validated_data.pop('image_id', None)
        doctor = self.context.get('doctor')

        for attr, value in validated_data.items():
            setattr(instance, attr, value)

        if image_ids is not None:
            current_images = set({image.id for image in instance.image.all()})
            new_images = set(image_ids)

            images_to_remove = current_images - new_images

            if images_to_remove:
                images_to_remove = File.objects.filter(id__in=images_to_remove)
                instance.image.remove(*images_to_remove)
                for image in images_to_remove:
                    if not DoctorCertificate.objects.filter(image=image).exists():
                        image.delete()

            images_to_add = new_images - current_images

            if images_to_add:
                for image_id in images_to_add:
                    image = File.objects.get(id=image_id)
                    if image.user != doctor.user:
                        raise serializers.ValidationError(f"Invalid image id {image.id}")
                    instance.image.add(image)

        instance.save()
        return instance


class DoctorAwardSerializer(serializers.ModelSerializer):
    image_id = serializers.ListField(child=serializers.IntegerField(), write_only=True)
    image_url = serializers.SerializerMethodField(read_only=True)
    image_ids = serializers.SerializerMethodField(read_only=True)

    class Meta:
        model = DoctorAward
        fields = ['id', 'doctor', 'title', 'dec', 'created_at', 'updated_at', 'image_id', 'image_url', 'image_ids']
        extra_kwargs = {
            'doctor': {'read_only': True}
        }

    def get_image_url(self, obj):
        return [image.file.url for image in obj.image.all()]

    def get_image_ids(self, obj):
        return [image.id for image in obj.image.all()]

    def create(self, validated_data):
        image_ids = validated_data.pop('image_id', [])
        images = File.objects.filter(id__in=image_ids)
        doctor = self.context.get('doctor')

        for image in images:
            if image.user != doctor.user:
                raise serializers.ValidationError(f"Invalid image id {image.id}")

        award = DoctorAward.objects.create(doctor=doctor, **validated_data)
        award.image.set(images)

        return award

    def update(self, instance, validated_data):

        image_ids = validated_data.pop('image_id', None)
        doctor = self.context.get('doctor')

        for attr, value in validated_data.items():
            setattr(instance, attr, value)

        if image_ids is not None:
            current_images = set({image.id for image in instance.image.all()})
            new_images = set(image_ids)

            images_to_remove = current_images - new_images

            if images_to_remove:
                images_to_remove = File.objects.filter(id__in=images_to_remove)
                instance.image.remove(*images_to_remove)
                for image in images_to_remove:
                    if not DoctorAward.objects.filter(image=image).exists():
                        image.delete()

            images_to_add = new_images - current_images

            if images_to_add:
                for image_id in images_to_add:
                    image = File.objects.get(id=image_id)
                    if image.user != doctor.user:
                        raise serializers.ValidationError(f"Invalid image id {image.id}")
                    instance.image.add(image)

        instance.save()
        return instance


class DoctorPrivateProfileSerializer(serializers.ModelSerializer):
    profile_image_id = serializers.CharField()
    profile_image_url = serializers.SerializerMethodField()
    mobile_phone = serializers.SerializerMethodField()
    medical_system_code = serializers.SerializerMethodField()
    workrecords = WorkRecordSerializer(many=True, required=False)
    educationhistories = EducationHistorySerializer(many=True, required=False)
    certificates = DoctorCertificateSerializer(many=True, required=False)
    first_name = serializers.CharField(source='user.first_name')
    last_name = serializers.CharField(source='user.last_name')
    email = serializers.EmailField(source='user.email')
    awards = DoctorAwardSerializer(many=True, required=False)


    class Meta:
        model = Doctor
        fields = ['id', 'first_name', 'last_name', 'email', 'birthday', 'expertise', 'national_code', 'medical_system_code', 'mobile_phone'
            , 'doctor_dec', 'profile_image_id', 'profile_image_url', 'created_at', 'updated_at','workrecords',
                  'educationhistories', 'certificates', 'awards', 'is_verified',]
        extra_kwargs = {
            'mobile_phone': {'read_only': True}, 'profile_image_url': {'read_only': True},
            'medical_system_code': {'read_only': True}, 'is_verified': {'read_only': True},
        }

    def get_profile_image_url(self, obj):
        if obj.profile_image:
            return obj.profile_image.file.url
        return None

    def get_mobile_phone(self, obj):
        return obj.user.mobile_phone if obj.user else None

    def get_medical_system_code(self, obj):
        return obj.medical_system_code if obj.user else None

    def update(self, instance, validated_data):
        user_data = validated_data.pop('user', {})
        for attr, value in validated_data.items():
            if attr not in ['profile_image_id',]:
                setattr(instance, attr, value)
        
        user = instance.user
        
        first_name = user_data.get('first_name')
        if first_name:
            user.first_name = first_name
        last_name = user_data.get('last_name')
        if last_name:
            user.last_name = last_name
        email = user_data.get('email')
        if email:
            user.email = email
        user.save()
        image_id = validated_data.get('profile_image_id')
        if image_id:
            try:
                new_image = File.objects.get(id=image_id)
                if new_image.user != self.context['request'].user:
                    raise PermissionDenied("کاربر اجازه بارگذاری عکس را ندارد")

                if instance.profile_image and instance.profile_image.id != new_image.id:
                    old_image = instance.profile_image
                    instance.profile_image = new_image
                    old_image.delete()
                elif not instance.profile_image:
                    instance.profile_image = new_image
            except File.DoesNotExist:
                raise PermissionDenied("چنین عکسی موجود نمی باشد")
        instance.name = user.get_full_name()
        instance.save()
        return instance

class DoctorPublicProfileSerializer(serializers.ModelSerializer):
    accepted_appointments_count = serializers.SerializerMethodField()
    appointments_count = serializers.SerializerMethodField()
    mobile_phone = serializers.CharField(source='user.mobile_phone', read_only=True)
    first_name = serializers.CharField(source='user.first_name', read_only=True)
    last_name = serializers.CharField(source='user.last_name', read_only=True)
    profile_image_url = serializers.SerializerMethodField()
    workrecords = WorkRecordSerializer(many=True)
    educationhistories = EducationHistorySerializer(many=True)
    certificates = DoctorCertificateSerializer(many=True)
    awards = DoctorAwardSerializer(many=True)

    class Meta:
        model = Doctor
        fields = ['id', 'first_name', 'last_name', 'profile_image_url', 'expertise',
                  'mobile_phone', 'medical_system_code', 'doctor_dec', 'workrecords',
                  'educationhistories', 'certificates', 'awards', 'is_verified', \
                    'accepted_appointments_count', 'appointments_count']
    def get_profile_image_url(self, obj):
        if obj.profile_image:
            return obj.profile_image.file.url
        return None

    def get_accepted_appointments_count(self, instance):
        try:
            count = PatientAppointment.objects.filter(doctor=instance, accepted=True).count()
        except:
            count = 0
        return count
    def get_appointments_count(self, instance):
        try:
            count = PatientAppointment.objects.filter(doctor=instance).count()
        except:
            count = 0
        return count

class DoctorListSerializer(serializers.ModelSerializer):
    profile_image_url = serializers.SerializerMethodField()
    first_name = serializers.CharField(source='user.first_name', read_only=True)
    last_name = serializers.CharField(source='user.last_name', read_only=True)

    class Meta:
        model = Doctor
        fields = ['id', 'first_name', 'last_name', 'is_verified', 'profile_image_url', 'medical_system_code', 'doctor_dec', 'expertise', 'created_at']
    
    def get_profile_image_url(self, obj):
        if obj.profile_image:
            return obj.profile_image.file.url
        return None

class DoctorPatientsSerializer(serializers.ModelSerializer):
    profile_image_url = serializers.SerializerMethodField()
 
    class Meta:
        model = Patient
        fields = ["id", 'user', 'profile_image_url', 'full_name',]

    def get_profile_image_url(self, obj):
        if obj.profile_image:
            return obj.profile_image.file.url
        return None
