from django.db.models import Q
from django.utils.translation import gettext as _
from rest_framework.views import APIView
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework.response import Response
from rest_framework import status
from rest_framework_simplejwt.tokens import RefreshToken
from apps.core.api.helpers import custom_get_or_404
from apps.doctor.permissions import IsDoctor, IsVerifiedDoctor
from apps.doctor.models import Doctor, DoctorWorkRecord, DoctorEducationHistory, DoctorAward, DoctorCertificate
from apps.patient.models import Patient, Dossier
from .serializers import (SignUp_LoginSerializer, OTPReferralVerificationSerializer, 
                          OTPVerificationSerializer, DoctorPrivateProfileSerializer,
                          DoctorPublicProfileSerializer, DoctorListSerializer, WorkRecordSerializer,
                          EducationHistorySerializer, DoctorAwardSerializer, DoctorCertificateSerializer,
                          DoctorPatientsSerializer,
                        )

class SignUp_Login(APIView):
    permission_classes = [AllowAny]

    def post(self, request):
        serializer = SignUp_LoginSerializer(data=request.data)

        if serializer.is_valid():
            serializer.save()
            if serializer.validated_data['user_is_new']:
                return Response({"message": _("کد یکبار مصرف و کد معرف و کد نظام پزشکی را وارد کنید")},
                                status=status.HTTP_201_CREATED)
            else:
                return Response({"message": _("کد یکبار مصرف را وارد کنید")}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class VerifyOtpReferralAPIView(APIView):
    permission_classes = [AllowAny]

    def post(self, request, *args, **kwargs):
        serializer = OTPReferralVerificationSerializer(data=request.data, context={'request': request})
        if serializer.is_valid():
            serializer.save()
            # generate an access token for the user
            doctor = serializer.validated_data['doctor']

            refresh_token = RefreshToken.for_user(user=doctor.user)

            return Response({"message": (".شما با موفقیت ثبت نام شدید"), 'access': str(refresh_token.access_token),
                             'refresh': str(refresh_token)}, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class UserVerifyCodeAPIView(APIView):
    permission_classes = [AllowAny]

    def post(self, request, *args, **kwargs):
        serializer = OTPVerificationSerializer(data=request.data, context={'request': request})
        if serializer.is_valid():
            serializer.save()
            # Generate a token for the user
            user = serializer.validated_data['user']

            refresh_token = RefreshToken.for_user(user=user)

            return Response({"message": ("شما با موفقیت وارد حساب خود شدید"), 'access': str(refresh_token.access_token),
                             'refresh': str(refresh_token)}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class PrivateProfileAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request):
        doctor = request.user.doctor
        serializer = DoctorPrivateProfileSerializer(doctor)
        return Response(serializer.data, status=status.HTTP_200_OK)

    def put(self, request):
        doctor = request.user.doctor
        serializer = DoctorPrivateProfileSerializer(doctor, data=request.data, context={'request': self.request})
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def patch(self, request):
        doctor = request.user.doctor
        serializer = DoctorPrivateProfileSerializer(doctor, data=request.data, context={'request': self.request},
                                                    partial=True)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class PublicProfileAPIView(APIView):
    permission_classes = [AllowAny]

    def get(self, request, doctor_id):
        doctor = custom_get_or_404(Doctor, id=doctor_id)
        serializer = DoctorPublicProfileSerializer(doctor)
        return Response(serializer.data, status=status.HTTP_200_OK)

class DoctorSearchAPIView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request, *args, **kwargs):
        search_param = request.query_params.get('search', None)
        expertise_lookup = request.query_params.get('expertise', None)
        queryset = Doctor.objects.filter(is_verified=True).exclude(user=request.user).order_by('-created_at')
        if search_param:
            queryset = queryset.filter(Q(name__icontains=search_param) | Q(medical_system_code__icontains=search_param))
        if expertise_lookup:
            queryset = queryset.filter(expertise__icontains=expertise_lookup)
        serializer = DoctorListSerializer(queryset, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)

class WorkRecordAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request):
        doctor = request.user.doctor
        work_records = DoctorWorkRecord.objects.filter(doctor=doctor)
        serialized_data = WorkRecordSerializer(work_records, many=True).data
        return Response(serialized_data, status=status.HTTP_200_OK)

    def post(self, request):
        doctor = request.user.doctor
        work_records = request.data.get('work_records', [])
        list_work_record = []

        for work in work_records:
            serializer = WorkRecordSerializer(data=work, context={'doctor': doctor})

            if serializer.is_valid():
                work_record = serializer.save()

                list_work_record.append(WorkRecordSerializer(work_record).data)
            else:
                return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        return Response({"work_records": list_work_record}, status=status.HTTP_201_CREATED)


class WorkRecordDetailAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request, *args, **kwargs):

        work_record_id = kwargs.get('work_record_id')
        doctor = request.user.doctor
        try:
            work_record = DoctorWorkRecord.objects.get(id=work_record_id, doctor=doctor)
        except DoctorWorkRecord.DoesNotExist:
            return Response({"error": "Work record not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = WorkRecordSerializer(work_record)
        return Response(serializer.data, status=status.HTTP_200_OK)

    def put(self, request, *args, **kwargs):

        work_record_id = kwargs.get('work_record_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            work_record = DoctorWorkRecord.objects.get(id=work_record_id, doctor=doctor)
        except DoctorWorkRecord.DoesNotExist:
            return Response({"error": "Work record not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = WorkRecordSerializer(work_record, data=request.data, context={'doctor': doctor})

        if serializer.is_valid():
            updated_work_record = serializer.save()
            return Response(
                WorkRecordSerializer(updated_work_record).data, status=status.HTTP_200_OK)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def patch(self, request, *args, **kwargs):
        work_record_id = kwargs.get('work_record_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            work_record = DoctorWorkRecord.objects.get(id=work_record_id, doctor=doctor)
        except DoctorWorkRecord.DoesNotExist:
            return Response({"error": "Work record not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = WorkRecordSerializer(work_record, data=request.data, partial=True, context={'doctor': doctor})

        if serializer.is_valid():
            updated_work_record = serializer.save()

            return Response(WorkRecordSerializer(updated_work_record).data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def delete(self, request, *args, **kwargs):
        work_record_id = kwargs.get('work_record_id')
        try:
            work_record = DoctorWorkRecord.objects.get(
                id=work_record_id,
                doctor__user=request.user
            )
        except DoctorWorkRecord.DoesNotExist:
            return Response({"error": "Work record not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        work_record.delete()
        return Response({"message": "Work record deleted successfully."}, status=status.HTTP_200_OK)


class EducationHistoryAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request):
        try:
            doctor = Doctor.objects.get(user=request.user)

        except Doctor.DoesNotExist:
            return Response({"error": "Doctor not found."}, status=status.HTTP_404_NOT_FOUND)

        education_histories = DoctorEducationHistory.objects.filter(doctor=doctor)
        serialized_data = EducationHistorySerializer(education_histories, many=True).data
        return Response({"education_histories": serialized_data}, status=status.HTTP_200_OK)

    def post(self, request):
        try:
            doctor = Doctor.objects.get(user=request.user)
        except Doctor.DoesNotExist:
            return Response({"error": "doctor not found."}, status=status.HTTP_404_NOT_FOUND)

        education_histories = request.data.get('education_histories', [])
        list_education_history = []

        for education in education_histories:
            serializer = EducationHistorySerializer(data=education, context={'doctor': doctor})

            if serializer.is_valid():
                education_history = serializer.save()

                list_education_history.append(EducationHistorySerializer(education_history).data)
            else:
                return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        return Response({"education_history": list_education_history}, status=status.HTTP_201_CREATED)


class EducationHistoryDetailAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request, *args, **kwargs):
        education_history_id = kwargs.get('education_history_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            education_history = DoctorEducationHistory.objects.get(id=education_history_id, doctor=doctor)
        except DoctorEducationHistory.DoesNotExist:
            return Response({"error": "Education history not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = EducationHistorySerializer(education_history)
        return Response(serializer.data, status=status.HTTP_200_OK)

    def put(self, request, *args, **kwargs):
        education_history_id = kwargs.get('education_history_id')

        try:
            doctor = Doctor.objects.get(user=request.user)
            education_history = DoctorEducationHistory.objects.get(id=education_history_id, doctor=doctor)
        except DoctorEducationHistory.DoesNotExist:
            return Response(
                {"error": "Education history not found or does not belong to the doctor."},
                status=status.HTTP_404_NOT_FOUND
            )

        serializer = EducationHistorySerializer(education_history, data=request.data, context={'doctor': doctor})

        if serializer.is_valid():
            updated_education_history = serializer.save()
            return Response(
                EducationHistorySerializer(updated_education_history).data,
                status=status.HTTP_200_OK
            )

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def patch(self, request, *args, **kwargs):
        education_history_id = kwargs.get('education_history_id')

        try:
            doctor = Doctor.objects.get(user=request.user)
            education_history = DoctorEducationHistory.objects.get(id=education_history_id, doctor=doctor)

        except DoctorEducationHistory.DoesNotExist:
            return Response({"error": "Education history not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = EducationHistorySerializer(education_history, data=request.data, partial=True,
                                                context={'doctor': doctor})

        if serializer.is_valid():
            updated_education_history = serializer.save()

            return Response(EducationHistorySerializer(updated_education_history).data,
                            status=status.HTTP_200_OK)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def delete(self, request, *args, **kwargs):
        education_history_id = kwargs.get('education_history_id')
        try:
            education_history = DoctorEducationHistory.objects.get(
                id=education_history_id,
                doctor__user=request.user
            )
        except DoctorEducationHistory.DoesNotExist:
            return Response({"error": "education history  not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        education_history.delete()
        return Response({"message": "education history deleted successfully."}, status=status.HTTP_200_OK)


class CertificateAPIView(APIView):
    permission_classes = [IsDoctor]
    serializer_class = DoctorCertificateSerializer

    def get(self, request):
        try:
            doctor = Doctor.objects.get(user=request.user)

        except Doctor.DoesNotExist:
            return Response({"error": "Doctor not found."}, status=status.HTTP_404_NOT_FOUND)

        certificates = DoctorCertificate.objects.filter(doctor=doctor)
        serialized_data = DoctorCertificateSerializer(certificates, many=True).data
        return Response({"certificates": serialized_data}, status=status.HTTP_200_OK)

    def post(self, request):
        try:
            doctor = Doctor.objects.get(user=request.user)
        except Doctor.DoesNotExist:
            return Response({"error": "doctor not found."}, status=status.HTTP_404_NOT_FOUND)

        certificates = request.data.get('certificates', [])
        list_certificate = []

        for cert in certificates:
            serializer = DoctorCertificateSerializer(data=cert, context={'doctor': doctor})

            if serializer.is_valid():
                certificate = serializer.save()

                list_certificate.append(DoctorCertificateSerializer(certificate).data)
            else:
                return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        return Response({"certificate": list_certificate}, status=status.HTTP_201_CREATED)


class CertificateDetailAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request, *args, **kwargs):
        certificate_id = kwargs.get('certificate_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            certificate = DoctorCertificate.objects.get(id=certificate_id, doctor=doctor)
        except DoctorCertificate.DoesNotExist:
            return Response({"error": "Certificate not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = DoctorCertificateSerializer(certificate)
        return Response(serializer.data, status=status.HTTP_200_OK)

    def put(self, request, *args, **kwargs):
        certificate_id = kwargs.get('certificate_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            certificate = DoctorCertificate.objects.get(id=certificate_id, doctor=doctor)
        except DoctorCertificate.DoesNotExist:
            return Response({"error": "Certificate not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = DoctorCertificateSerializer(certificate, data=request.data, context={'doctor': doctor})

        if serializer.is_valid():
            updated_certificate = serializer.save()
            return Response(DoctorCertificateSerializer(updated_certificate).data, status=status.HTTP_200_OK)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def patch(self, request, *args, **kwargs):
        certificate_id = kwargs.get('certificate_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            certificate = DoctorCertificate.objects.get(id=certificate_id, doctor=doctor)
        except DoctorCertificate.DoesNotExist:
            return Response({"error": "Certificate not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = DoctorCertificateSerializer(certificate, data=request.data, partial=True,
                                                 context={'doctor': doctor})

        if serializer.is_valid():
            updated_certificate = serializer.save()

            return Response(DoctorCertificateSerializer(updated_certificate).data,
                            status=status.HTTP_200_OK)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def delete(self, request, *args, **kwargs):
        certificate_id = kwargs.get('certificate_id')
        try:
            certificate = DoctorCertificate.objects.get(
                id=certificate_id,
                doctor__user=request.user
            )
        except DoctorCertificate.DoesNotExist:
            return Response({"error": "certificate not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        certificate.delete()
        return Response({"message": "certificate deleted successfully."}, status=status.HTTP_200_OK)


class AwardAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request):
        try:
            doctor = Doctor.objects.get(user=request.user)

        except Doctor.DoesNotExist:
            return Response({"error": "Doctor not found."}, status=status.HTTP_404_NOT_FOUND)

        awards = DoctorAward.objects.filter(doctor=doctor)
        serialized_data = DoctorAwardSerializer(awards, many=True).data
        return Response({"awards": serialized_data}, status=status.HTTP_200_OK)

    def post(self, request):
        try:
            doctor = Doctor.objects.get(user=request.user)
        except Doctor.DoesNotExist:
            return Response({"error": "doctor not found."}, status=status.HTTP_404_NOT_FOUND)

        awards = request.data.get('awards', [])
        list_award = []

        for award in awards:
            serializer = DoctorAwardSerializer(data=award, context={'doctor': doctor})

            if serializer.is_valid():
                award = serializer.save()

                list_award.append(DoctorAwardSerializer(award).data)
            else:
                return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        return Response({"award": list_award}, status=status.HTTP_201_CREATED)


class AwardDetailAPIView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request, *args, **kwargs):
        award_id = kwargs.get('award_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            award = DoctorAward.objects.get(id=award_id, doctor=doctor)
        except DoctorAward.DoesNotExist:
            return Response({"error": "Award not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = DoctorAwardSerializer(award)
        return Response(serializer.data, status=status.HTTP_200_OK)

    def put(self, request, *args, **kwargs):

        award_id = kwargs.get('award_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            award = DoctorAward.objects.get(id=award_id, doctor=doctor)
        except DoctorAward.DoesNotExist:
            return Response({"error": "Award not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = DoctorAwardSerializer(award, data=request.data, context={'doctor': doctor})

        if serializer.is_valid():
            updated_award = serializer.save()
            return Response(DoctorAwardSerializer(updated_award).data, status=status.HTTP_200_OK)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def patch(self, request, *args, **kwargs):

        award_id = kwargs.get('award_id')
        try:
            doctor = Doctor.objects.get(user=request.user)
            award = DoctorAward.objects.get(id=award_id, doctor=doctor)
        except DoctorAward.DoesNotExist:
            return Response({"error": "Award not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        serializer = DoctorAwardSerializer(award, data=request.data, partial=True, context={'doctor': doctor})

        if serializer.is_valid():
            updated_award = serializer.save()

            return Response(DoctorAwardSerializer(updated_award).data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def delete(self, request, *args, **kwargs):
        award_id = kwargs.get('award_id')
        try:
            award = DoctorAward.objects.get(
                id=award_id,
                doctor__user=request.user
            )
        except DoctorAward.DoesNotExist:
            return Response({"error": "award not found or does not belong to the doctor."},
                            status=status.HTTP_404_NOT_FOUND)

        award.delete()
        return Response({"message": "award deleted successfully."}, status=status.HTTP_200_OK)

class DoctorPatientsListView(APIView):
    permission_classes = [IsDoctor]

    def get(self, request, *args, **kwargs):
        doctor = request.user.doctor
        patient_ids = Dossier.objects.filter(doctor=doctor).values_list('patient__id', flat=True)
        patients = Patient.objects.filter(id__in=patient_ids)
        serializer = DoctorPatientsSerializer(patients, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)