import uuid
from django.db import models
from django.contrib.auth import get_user_model

User = get_user_model()

def generic_image_upload_to(instance, filename):
    # Generate a unique filename
    extension = filename.split('.')[-1]
    new_filename = f"{uuid.uuid4()}.{extension}"
    return f"images/{new_filename}"

class Doctor(models.Model):
    user = models.OneToOneField(User, on_delete=models.PROTECT, related_name='doctor')
    invited_by_code = models.CharField(max_length=10, blank=True, null=True)
    identification_code = models.CharField(max_length=10, blank=True, null=True)
    is_verified = models.BooleanField(default=False)
    # TODO: Delete name field for v1.0 - Use user.get_full_name() instead
    name = models.CharField(max_length=120, blank=True, null=True)
    profile_image = models.ForeignKey('core.File', on_delete=models.CASCADE, null=True, blank=True)
    # TODO: Delete email field for v1.0 - Use user.email instead
    email = models.EmailField(blank=True, null=True)
    birthday = models.DateField(blank=True, null=True)
    expertise = models.CharField(max_length=120, blank=True, null=True)
    national_code = models.CharField(max_length=10, blank=True, null=True)
    medical_system_code = models.CharField(max_length=15, blank=False, null=False)
    doctor_dec = models.TextField(blank=True, null=True)
    visit_time =models.IntegerField(default=15)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.user.mobile_phone
    
    def initialize(self):
        self.visit_time = 15
        self.is_verified = True
        self.save()

class DoctorCertificate(models.Model):
    doctor = models.ForeignKey(Doctor, related_name='certificates', on_delete=models.CASCADE)
    image = models.ManyToManyField('core.File')
    title = models.CharField(max_length=30)
    dec = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ("-created_at",)

    def __str__(self):
        return self.doctor.id


class DoctorWorkRecord(models.Model):
    doctor = models.ForeignKey(Doctor, related_name='workrecords', on_delete=models.CASCADE)
    title = models.CharField(max_length=20)
    dec = models.TextField()
    is_current = models.BooleanField(default=False)
    from_date = models.DateField()
    to_date = models.DateField(null=True, blank=True)
    image = models.ManyToManyField('core.File', blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ("-created_at",)

    def __str__(self):
        return self.title


class DoctorEducationHistory(models.Model):
    doctor = models.ForeignKey(Doctor, related_name='educationhistories', on_delete=models.CASCADE)
    title = models.CharField(max_length=30, blank=True, null=True)
    dec = models.TextField()
    from_date = models.DateField()
    to_date = models.DateField(null=True, blank=True)
    image = models.ManyToManyField('core.File', blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ("-created_at",)

    def __str__(self):
        return self.title


class DoctorAward(models.Model):
    doctor = models.ForeignKey(Doctor, related_name='awards', on_delete=models.CASCADE)
    title = models.CharField(max_length=30)
    dec = models.TextField(null=True, blank=True)
    image = models.ManyToManyField('core.File', blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ("-created_at",)

    def __str__(self):
        return self.title


class Clinic(models.Model):
    doctor = models.ForeignKey(Doctor, related_name="clinic", on_delete=models.CASCADE)
    title = models.CharField(max_length=120)
    address = models.TextField()
    latitude = models.DecimalField(max_digits=9, decimal_places=6)
    longitude = models.DecimalField(max_digits=9, decimal_places=6)
    phone = models.CharField(max_length=20)
    created_at = models.DateTimeField(auto_now_add=True)
    upated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ("-created_at",)

    def __str__(self):
        return self.title
