from django.utils.translation import gettext_lazy as _
from rest_framework import serializers
from rest_framework.exceptions import PermissionDenied
from apps.core.models import File
from apps.user.serializers import SignUp_LoginSerializer as UserAuthSerializer, OTPVerificationSerializer as UserOTPSerializer
from apps.doctor.models import Doctor
from apps.doctor.api.v1.serializers import DoctorPublicProfileSerializer
from apps.patient.models import Patient, Dossier, Visit, Medicine, History, TherapeuticNotice, DossierTest, DossierNoteHistory


class SignUp_LoginSerializer(UserAuthSerializer):
    USER_DETAIL_MODEL = Patient

class OTPVerificationSerializer(UserOTPSerializer):
    USER_TYPE = 'patient'

class PatientProfileSerializer(serializers.ModelSerializer):
    user = serializers.CharField(read_only=True)
    profile_image_id = serializers.CharField()
    profile_image_url = serializers.SerializerMethodField()
    visits_count = serializers.SerializerMethodField()
    referral_count = serializers.SerializerMethodField()
    dossier_count = serializers.SerializerMethodField()
    # full_name = serializers.SerializerMethodField()
    email = serializers.EmailField(source='user.email')
    first_name = serializers.CharField(source='user.first_name')
    last_name = serializers.CharField(source='user.last_name')

    class Meta:
        model = Patient
        fields = ['id', 'user', 'profile_image_url', 'profile_image_id', 'full_name', 'email', 'birth_date',\
                   'father_name', 'national_code', 'membership_code',\
                   'gender', 'blood_type', 'created_at', 'updated_at', \
                    'visits_count', 'referral_count', 'dossier_count', 'first_name', 'last_name']
        extra_kwargs = {
            "membership_code": {"read_only": True},
            "user": {"read_only": True},
            "full_name": {"read_only": True},
        }
    def get_profile_image_url(self, obj):
        if obj.profile_image:
            return obj.profile_image.file.url
        return None
    
    def get_visits_count(self, instance):
        return Visit.objects.filter(dossier__patient=instance).count()

    def get_referral_count(self, instance):
        return 0
    
    def get_dossier_count(self, instance):
        return Dossier.objects.filter(patient=instance).count() - 1
    
    def validate(self, data, *args, **kwargs):
        image_id = data.get('profile_image_id')
        if image_id:
            try:
                new_image = File.objects.get(id=image_id)
                if new_image.user != self.context['request'].user:
                    raise serializers.ValidationError("کاربر اجازه بارگذاری عکس را ندارد")
            except File.DoesNotExist:
                raise serializers.ValidationError("چنین عکسی موجود نمی باشد")
        return super().validate(data, *args, **kwargs)

    def update(self, instance, validated_data):
        user_data = validated_data.pop('user', {})
        user = instance.user
        for attr, value in validated_data.items():
            if attr not in ['profile_image_id']:
                setattr(instance, attr, value)
                first_name = user_data.get('first_name')
        if first_name:
            user.first_name = first_name
        last_name = user_data.get('last_name')
        if last_name:
            user.last_name = last_name
        email = user_data.get('email')
        if email:
            user.email = email
        user.save()
        instance.full_name = f"{user.first_name} {user.last_name}"
        image_id = validated_data.get('profile_image_id')
        if image_id:
            try:
                new_image = File.objects.get(id=image_id)
                if new_image.user != self.context['request'].user:
                    raise PermissionDenied("کاربر اجازه بارگذاری عکس را ندارد")

                if instance.profile_image and instance.profile_image.id != new_image.id:
                    old_image = instance.profile_image
                    instance.profile_image = new_image
                    old_image.delete()
                elif not instance.profile_image:
                    instance.profile_image = new_image
            except File.DoesNotExist:
                raise PermissionDenied("چنین عکسی موجود نمی باشد")
        instance.save()
        return instance

class MedicineSerializer(serializers.ModelSerializer):
    class Meta:
        model = Medicine
        exclude = ["created_at"]
        extra_kwargs = {
            "dossier": {'read_only': True}
        }


class HistorySerializer(serializers.ModelSerializer):
    class Meta:
        model = History
        fields = ['dossier', "id", "note"]
        extra_kwargs = {
            "dossier": {"read_only": True}
        }


class TherapeuticNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = TherapeuticNotice
        fields = ["dossier", "id", "therapeutic_notice"]
        extra_kwargs = {
            "dossier": {"read_only": True}
        }


class DossierTestSerializer(serializers.ModelSerializer):
    image_id = serializers.ListField(
        child=serializers.IntegerField(), write_only=True)
    image_url = serializers.SerializerMethodField(read_only=True)


    class Meta:
        model = DossierTest
        exclude = ["created_at", "updated_at",]
        extra_kwargs = {
            "dossier": {"read_only": True}
        }

    def get_image_url(self, obj):
        image = [i.file.url if obj.image else None for i in obj.image.all()]
        return image


class DossierNoteHistorySerializer(serializers.ModelSerializer):
    doctor = DoctorPublicProfileSerializer(required=False)
    class Meta:
        model = DossierNoteHistory
        fields = '__all__'
        extra_kwargs = {
            'dossier': {'read_only': True},
            "doctor": {"read_only": True}
        }


class DossierPatientSerializer(serializers.ModelSerializer):
    user = serializers.CharField()
    profile_image_url = serializers.SerializerMethodField()

    class Meta:
        model = Patient
        fields = ['id', 'user', "full_name", "national_code", "birth_date", "gender", "blood_type", "profile_image_url"]

    def get_profile_image_url(self, obj):
        if obj.profile_image:
            return obj.profile_image.file.url
        return None


class DoctorDossierSerializer(serializers.ModelSerializer):
    doctor = DoctorPublicProfileSerializer(required=False)
    disease_history = HistorySerializer(many=True, required=False)
    therapeutic_notice = TherapeuticNoteSerializer(many=True, required=False)
    dossier_test_dossier = DossierTestSerializer(many=True, required=False)
    medicine = MedicineSerializer(many=True, required=False)
    dossier_note_history = DossierNoteHistorySerializer(many=True, required=False)
    patient = DossierPatientSerializer(required=False)

    class Meta:
        model = Dossier
        fields = '__all__'

    def validate(self, attrs):
        if "patient" in attrs:
            patient = attrs["patient"]
            try:
                p = Patient.objects.get(user__mobile_phone__exact=patient['user'])
            except Patient.DoesNotExist:
                raise serializers.ValidationError({"patient": _("چنین بیماری وجود ندارد")})
            attrs['valid_patient'] = p
        return attrs

    def create(self, validated_data):
        # Because we have already checked it in the validation method, we will remove it
        del validated_data['patient']['user']
        # get doctor in request
        doctor = Doctor.objects.get(user=self.context['request'].user)
        # get valid_patient
        valid_patient = validated_data['valid_patient']

        if Dossier.objects.filter(doctor=doctor, patient=valid_patient).exists():
            raise serializers.ValidationError({"message": "پرونده‌ای برای این بیمار موجود است."})
        dossier = Dossier.objects.create(doctor=doctor, patient=valid_patient)

        medicine = validated_data.pop('medicine', [])
        for m in medicine:
            Medicine.objects.create(dossier=dossier, **m)

        disease_history = validated_data.pop('disease_history', [])
        for d in disease_history:
            History.objects.create(dossier=dossier, **d)

        therapeutic_notice = validated_data.pop('therapeutic_notice', [])
        for t in therapeutic_notice:
            TherapeuticNotice.objects.create(dossier=dossier, **t)

        tests = validated_data.pop('dossier_test_dossier', [])
        for test in tests:
            image_id = test.pop('image_id', None)  # Get the image ID for the test record
            try:
                test_images_instance = File.objects.filter(id__in=image_id)
            except File.DoesNotExist:
                    raise serializers.ValidationError(f"شناسه فایل نامعتبر است.")
            for image in test_images_instance:
                if image.user != doctor.user:
                    raise PermissionDenied(f"Image with ID {image.id} does not belong to the current user.")
     
                
            dossier_test=DossierTest.objects.create(dossier=dossier, **test)
            dossier_test.image.set(test_images_instance)
            

        note = validated_data.pop('dossier_note_history', [])
        for n in note:
            DossierNoteHistory.objects.create(dossier=dossier, **n, doctor=doctor)

        return dossier

    def update(self, instance, validated_data):
        # get validate_patient
        dossier = instance

        medicine = validated_data.pop('medicine', [])

        Medicine.objects.filter(dossier=dossier).delete()
        for m in medicine:
            Medicine.objects.create(dossier=dossier, **m)

        disease_history = validated_data.pop('disease_history', [])
        History.objects.filter(dossier=dossier).delete()
        for d in disease_history:
            History.objects.create(dossier=dossier, **d)

        therapeutic_notice = validated_data.pop('therapeutic_notice', [])
        TherapeuticNotice.objects.filter(dossier=dossier).delete()
        for t in therapeutic_notice:
            TherapeuticNotice.objects.create(dossier=dossier, **t)

        DossierTest.objects.filter(dossier=dossier).delete()
        tests = validated_data.pop('dossier_test_dossier', [])
        for test in tests:
            image_id = test.pop('image_id', None)  # Get the image ID for the test record
            try:
                test_images_instance = File.objects.filter(id__in=image_id)
            except File.DoesNotExist:
                    raise serializers.ValidationError(f"شناسه فایل نامعتبر است.")
            for image in test_images_instance:
                if image.user != self.context['request'].user:
                    raise PermissionDenied(f"Image with ID {image.id} does not belong to the current user.")
     
                
            dossier_test=DossierTest.objects.create(dossier=dossier, **test)
            dossier_test.image.set(test_images_instance)
            
        DossierNoteHistory.objects.filter(dossier=dossier).delete()
        note = validated_data.pop('dossier_note_history', [])
        for n in note:
            DossierNoteHistory.objects.create(dossier=dossier, **n, doctor=instance.doctor)

        return dossier

class UserDoctorSerializer(serializers.ModelSerializer):
    patient = PatientProfileSerializer()
    doctor = serializers.CharField()
    number_of_visits = serializers.SerializerMethodField()
    last_visit = serializers.SerializerMethodField()

    class Meta:
        model = Dossier
        fields = ["id", "patient", "doctor", "created_at", "updated_at", "number_of_visits", "last_visit"]
    
    def get_number_of_visits(self, instance):
        return Visit.objects.filter(dossier=instance).count()
    
    def get_last_visit(self, instance):
        last_visit = Visit.objects.filter(dossier=instance).order_by('-created_at').last()
        return last_visit.created_at if last_visit else None
    
class UserPatientDossier(serializers.ModelSerializer):
    doctor = DoctorPublicProfileSerializer()
    patient = serializers.CharField()
    number_of_visits = serializers.SerializerMethodField()
    last_visit = serializers.SerializerMethodField()

    class Meta:
        model = Dossier
        fields = ["id", "patient", "doctor", "created_at", "updated_at", "number_of_visits", "last_visit"]
    
    def get_number_of_visits(self, instance):
        return Visit.objects.filter(dossier=instance).count()
    
    def get_last_visit(self, instance):
        last_visit = Visit.objects.filter(dossier=instance).order_by('-created_at').last()
        return last_visit.created_at if last_visit else None

class DossierSerializer(serializers.ModelSerializer):
    patient = PatientProfileSerializer(read_only=True)
    doctor = DoctorPublicProfileSerializer(read_only=True)

    class Meta:
        model = Dossier
        fields = '__all__'


class VisitSerializer(serializers.ModelSerializer):
    dossier = DossierSerializer(required=False)
    dossier__patient = PatientProfileSerializer(read_only=True)
    dossier__doctor = DoctorPublicProfileSerializer(read_only=True)

    class Meta:
        model = Visit
        fields = ["id", 'dossier', "dossier__doctor", "dossier__patient", "created_at", "updated_at", ]
        read_only_fields=['dossier']

    def create(self, validated_data):
        visit = Visit.objects.create(dossier=validated_data.get('dossier'))
        return visit
