from django.db.models import Q
from django.utils.translation import gettext as _
from rest_framework.views import APIView
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework.response import Response
from rest_framework.generics import get_object_or_404
from rest_framework import status
from rest_framework_simplejwt.tokens import RefreshToken
from apps.core.api.helpers import custom_get_or_404
from apps.doctor.permissions import IsDoctor
from apps.patient.models import Patient, Dossier, Visit, TherapeuticNotice, DossierNoteHistory
from apps.patient.permissions import IsPatient
from .serializers import (SignUp_LoginSerializer, OTPVerificationSerializer, PatientProfileSerializer,
                          DoctorDossierSerializer, UserDoctorSerializer, UserPatientDossier, VisitSerializer,)
from .paginations import DossierPagination, PatientDossierPagination, DossierVisitPagination

class SignUp_Login(APIView):
    permission_classes = [AllowAny]

    def post(self, request):
        serializer = SignUp_LoginSerializer(data=request.data)

        if serializer.is_valid():
            serializer.save()
            status_code = None
            if serializer.validated_data['user_is_new']:
                status_code = status.HTTP_201_CREATED
            else:
                status_code = status.HTTP_200_OK
            return Response({"message": _("کد یکبار مصرف را وارد کنید")}, status=status_code)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class UserVerifyCodeAPIView(APIView):
    permission_classes = [AllowAny]

    def post(self, request, *args, **kwargs):
        serializer = OTPVerificationSerializer(data=request.data, context={'request': request})
        if serializer.is_valid():
            serializer.save()
            # Generate a token for the user
            user = serializer.validated_data['user']

            refresh_token = RefreshToken.for_user(user=user)

            return Response({"message": ("شما با موفقیت وارد حساب خود شدید"), 'access': str(refresh_token.access_token),
                             'refresh': str(refresh_token)}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class PatientApiView(APIView):
    serializer_class = PatientProfileSerializer
    queryset = Patient.objects.select_related('user')
    permission_classes = [IsPatient]

    def get_object(self):
        return get_object_or_404(self.queryset, user=self.request.user)

    def get(self, request, *args, **kwargs):
        query = self.get_object()
        serializer = self.serializer_class(query, context={'request': request})
        return Response(serializer.data, status=status.HTTP_200_OK)

    def put(self, request, *args, **kwargs):
        query = self.get_object()
        serializer = self.serializer_class(query, data=request.data, context={'request': request})
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return Response(serializer.data, status=status.HTTP_200_OK)

    def patch(self, request, *args, **kwargs):
        query = self.get_object()
        serializer = self.serializer_class(query, data=request.data, partial=True, context={'request': request})
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return Response(serializer.data, status=status.HTTP_200_OK)

class PatientsDefaultDossierAPIView(APIView):
    permission_classes = [IsPatient]

    def get(self, request, *args, **kwargs):
        patient = Patient.objects.get(user=request.user)
        dossier = Dossier.objects.filter(patient=patient, doctor__isnull=True).first()
        serializer = DoctorDossierSerializer(dossier)
        return Response(serializer.data, status=status.HTTP_200_OK)
    
    def put(self, request, *args, **kwargs):
        patient = Patient.objects.get(user=request.user)
        dossier = Dossier.objects.filter(patient=patient, doctor__isnull=True).first()
        request.data.pop('dossier_note_history')
        serializer = DoctorDossierSerializer(dossier, data=request.data, context={'request': request})
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class DossierSearchAPIView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request, *args, **kwargs):
        search_query = request.query_params.get('search', None)
        if not search_query:
            return Response({"error": "No search query"}, status=status.HTTP_400_BAD_REQUEST)
        queryset = Dossier.objects.all()
        is_doctor = None
        if request.user.is_doctor():
            queryset = queryset.filter(doctor__user=request.user)
            is_doctor = True
        elif request.user.is_patient():
            queryset = queryset.filter(patient__user=request.user)
            is_doctor = False
        else:
            return Response({"error": "Unauthorized user"}, status=status.HTTP_401_UNAUTHORIZED)
        try:
            search_query = int(search_query)
            queryset = queryset.filter(id__icontains=search_query)
        except:
            if is_doctor:
                queryset = queryset.filter(patient__full_name__icontains=search_query)
            else:
                queryset = queryset.filter(doctor__name__icontains=search_query)
        return Response(DoctorDossierSerializer(queryset, many=True).data, status=status.HTTP_200_OK)

class DossierApiview(APIView, DossierPagination):
    serializer_class = DoctorDossierSerializer
    queryset = (Dossier.objects.select_related("patient__user", "doctor__user").
                prefetch_related("history", "therapeutic_notice", "medicine", "dossier_note_history",
                                 "dossier_test_dossier__image"))

    def get_object(self, *args, **kwargs):
        if 'dossier_pk' in kwargs:
            return get_object_or_404(self.queryset.filter(Q(patient__user=self.request.user) | Q(doctor__user=self.request.user)), pk=kwargs['dossier_pk'])
        # return self.queryset.filter(patient__user=self.request.user)

    def get_permissions(self):
        if self.request.method in ["PUT", "PATCH", "POST", "DELETE"]:
            return [IsDoctor()]
        return [IsAuthenticated()]

    def get(self, request, *args, **kwargs):
        if 'dossier_pk' in kwargs:
            query = self.get_object(*args, **kwargs)
            serializer = self.serializer_class(query)
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response({"message": _("missing dossier_pk")}, status=status.HTTP_400_BAD_REQUEST)

    def post(self, request, *args, **kwargs):
        serializer = self.serializer_class(data=request.data, context={"request": request})
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return Response(serializer.data, status=status.HTTP_201_CREATED)

    def put(self, request, *args, **kwargs):
        if "dossier_pk" in kwargs:
            serializer = self.serializer_class(self.get_object(dossier_pk=kwargs['dossier_pk']), data=request.data)
            serializer.is_valid(raise_exception=True)
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response({"message": _("dossier id not provided")})

    def patch(self, request, *args, **kwargs):
        if "dossier_pk" in kwargs:
            serializer = self.serializer_class(self.get_object(dossier_pk=kwargs['dossier_pk']), data=request.data,
                                               partial=True)
            serializer.is_valid(raise_exception=True)
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response({"message": _("dossier id not provided")})

    def delete(self, request, *args, **kwargs):
        if "dossier_pk" in kwargs:
            query = self.get_object(*args, **kwargs)
            if not query.doctor:
                return Response({"message": "You can't delete default dossiers"}, status=status.HTTP_400_BAD_REQUEST)
            query.delete()
            return Response(status=status.HTTP_204_NO_CONTENT)
        return Response({"message": _("dossier id not provided")})

class UserDoctorDossierApiView(APIView, DossierPagination):
    serializer_class = UserDoctorSerializer
    permission_classes = [IsDoctor]
    queryset = Dossier.objects.select_related("patient__user", "doctor__user")

    def get(self, request, *args, **kwargs):
        query = self.queryset.filter(doctor__user=request.user)
        result = self.paginate_queryset(query, request, view=self)
        serializer = self.serializer_class(result, many=True)
        return self.get_paginated_response(serializer.data)

class UserPatientDossierApiView(APIView, PatientDossierPagination):
    serializer_class = UserPatientDossier
    permission_classes = [IsAuthenticated]

    def get(self, request, *args, **kwargs):
        query = Dossier.objects.select_related("doctor__user", "patient__user").filter(patient__user=request.user).exclude(doctor__isnull=True)
        result = self.paginate_queryset(query, request, view=self)
        serializer = self.serializer_class(result, many=True)
        return self.get_paginated_response(serializer.data)

class VisitDossierApiView(APIView, DossierVisitPagination):
    """
    show visit dossier api view
    """
    serializer_class = VisitSerializer
    queryset = Visit.objects.select_related("dossier__doctor__user", "dossier__patient__user")
    permission_classes = [IsDoctor]

    def get(self, request, *args, **kwargs):
        if hasattr(request.user, "doctor"):
            query = self.queryset.filter(dossier__doctor__user=request.user)
        else:
            return Response(status=status.HTTP_403_FORBIDDEN)
        result = self.paginate_queryset(query, request, view=self)
        serializer = self.serializer_class(result, many=True)
        return self.get_paginated_response(serializer.data)

class VisitApiView(APIView):
    serializer_class = VisitSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        dossier = get_object_or_404(Dossier, pk=self.kwargs['dossier_pk'])
        visit = Visit.objects.filter(dossier__id=dossier.id).select_related('dossier')
        return visit

    def get_object(self, *args, **kwargs):
        dossier = get_object_or_404(Dossier, pk=self.kwargs['dossier_pk'])
        visit = get_object_or_404(Visit, dossier=dossier, pk=kwargs['visit_pk'])
        return visit

    def get_permissions(self):
        if self.request.method in ["GET"]:
            return [IsAuthenticated()]
        return super().get_permissions()

    def get(self, request, *args, **kwargs):
        if "visit_pk" in kwargs:
            query = self.get_object(*args, **kwargs)
            serializer = self.serializer_class(query)
            return Response(serializer.data, status=status.HTTP_200_OK)
        query = self.get_queryset()
        serializer = self.serializer_class(query, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)

    def post(self, request, *args, **kwargs):
        if "visit_pk" in kwargs:
            return Response({"method": "method not allowed"}, status=status.HTTP_400_BAD_REQUEST)
        serializer = self.serializer_class(data=request.data, context={'request': request})
        serializer.is_valid(raise_exception=True)
        serializer.save(dossier = get_object_or_404(Dossier, pk=kwargs['dossier_pk']))
        return Response(serializer.data, status=status.HTTP_201_CREATED)

    def put(self, request, *args, **kwargs):
        if "visit_pk" in kwargs:
            query = self.get_object(*args, **kwargs)
            serializer = self.serializer_class(query, data=request.data)
            serializer.is_valid(raise_exception=True)
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)

    def patch(self, request, *args, **kwargs):
        if "visit_pk" in kwargs:
            query = self.get_object(*args, **kwargs)
            serializer = self.serializer_class(query, data=request.data, partial=True)
            serializer.is_valid(raise_exception=True)
            serializer.save()
            return Response(serializer.data, status=status.HTTP_200_OK)

    def delete(self, request, *args, **kwargs):
        if "visit_pk" in kwargs:
            query = self.get_object(*args, **kwargs)
            query.delete()
            return Response(status=status.HTTP_204_NO_CONTENT)

class DossierNotesAPIView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request, dossier_id, *args, **kwargs):
        dossier = get_object_or_404(Dossier, pk=dossier_id)
        notes = []
        therapeutic_notes = TherapeuticNotice.objects.filter(dossier=dossier)
        for tn in therapeutic_notes:
            notes.append({
                'id': tn.id,
                'note': tn.therapeutic_notice,
                'created_at': tn.created_at,
                'type': "therapeutic_notice"
            })
        history_notes = DossierNoteHistory.objects.filter(dossier=dossier)
        for hn in history_notes:
            notes.append({
                'id': hn.id,
                'note': hn.dossier_history_note,
                'created_at': hn.created_at,
                'type': "dossier_note_history"
            })
        
        notes.sort(key=lambda x: x['created_at'], reverse=True)

        return Response(notes, status=status.HTTP_200_OK)

class GetPatientPublicAPIView(APIView):
    def get(self, request, patient_id):
        patient = get_object_or_404(Patient, pk=patient_id)
        serialized_patient = PatientProfileSerializer(patient).data
        default_dossier = Dossier.objects.filter(patient=patient, doctor__isnull=True).first()
        serialized_dossier = DoctorDossierSerializer(default_dossier).data
        return Response({"patient": serialized_patient, "default_dossier": serialized_dossier})

