import random
import uuid
from django.db import models
from django.contrib.auth import get_user_model

User = get_user_model()

def generic_image_upload_to(instance, filename):
    # Generate a unique filename
    extension = filename.split('.')[-1]
    new_filename = f"{uuid.uuid4()}.{extension}"
    return f"images/patients/{new_filename}"

class Patient(models.Model):
    """
    patient profile
    """
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name='patient')
    profile_image = models.ForeignKey('core.File', on_delete=models.SET_NULL, null=True, blank=True)
    birth_date = models.DateField(blank=True, null=True)
    full_name = models.CharField(max_length=250, blank=True, null=True)
    father_name = models.CharField(max_length=120, blank=True, null=True)
    national_code = models.CharField(max_length=10, null=True, blank=True)
    membership_code = models.CharField(max_length=10, unique=True, editable=False)
    gender = models.CharField(max_length=6, blank=True, null=True)
    blood_type = models.CharField(max_length=8, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.user.mobile_phone

    def initialize(self):
        while True:
            code = str(random.randint(100000000, 999999999))
            if not Patient.objects.filter(membership_code=code).exists():
                break
        self.membership_code = code
        self.save()
        Dossier.objects.create(patient=self, doctor=None)

class Dossier(models.Model):
    """
    patient dossier
    """
    patient = models.ForeignKey(Patient, on_delete=models.CASCADE, related_name='dossier')
    doctor = models.ForeignKey('doctor.Doctor', on_delete=models.CASCADE, related_name='doctor_dossier', blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    @property
    def get_doctor_phone(self):
        if self.doctor:
            return self.doctor.user.mobile_phone
        return None

    @property
    def get_patient_phone(self):
        return self.patient.user.mobile_phone

    def __str__(self):
        return f'patient: {self.get_patient_phone} doctor: {self.get_doctor_phone}'

    class Meta:
        ordering = ('-created_at',)


class DossierTest(models.Model):
    """
    patient dossier tests
    """
    dossier = models.ForeignKey(Dossier, on_delete=models.CASCADE, related_name='dossier_test_dossier')
    title = models.CharField(max_length=250)
    date = models.DateField()
    image = models.ManyToManyField('core.File', related_name='test_image', blank=True)
    prescription = models.ForeignKey('patient.Prescription', on_delete=models.CASCADE, related_name='prescription_tests', blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f'title: {self.title} - dossier: {self.dossier_id}'


class History(models.Model):
    """
    patient's history
    """
    note = models.TextField()
    dossier = models.ForeignKey(Dossier, on_delete=models.CASCADE, related_name='history')
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{str(self.dossier_id)} - {self.created_at.strftime('%Y-%m-%d %H:%M:%S')}"


class Medicine(models.Model):
    """
    patient medicine
    """
    dossier = models.ForeignKey(Dossier, on_delete=models.CASCADE, related_name='medicine')
    medicine_name = models.CharField(max_length=250)
    start_date = models.DateField()
    end_date = models.DateField(blank=True, null=True)
    dosage = models.CharField(max_length=250, blank=True, null=True)
    prescription = models.ForeignKey('patient.Prescription', on_delete=models.CASCADE, related_name='prescription_medicine', blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f'{self.medicine_name} {self.dosage}'

class TherapeuticNotice(models.Model):
    """
    patient therapeutic notice
    """
    dossier = models.ForeignKey(Dossier, on_delete=models.CASCADE, related_name='therapeutic_notice')
    therapeutic_notice = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"{self.dossier_id} - {self.therapeutic_notice[:50]}"



class Visit(models.Model):
    dossier = models.ForeignKey(Dossier, on_delete=models.CASCADE, related_name='visit')
    updated_at = models.DateTimeField(auto_now=True)
    created_at = models.DateTimeField(auto_now_add=True)

class Prescription(models.Model):
    visit = models.ForeignKey(Visit, on_delete=models.CASCADE, related_name='prescription',)
    updated_at = models.DateTimeField(auto_now=True)
    created_at = models.DateTimeField(auto_now_add=True)

class PrescriptionDiagnosticNotes(models.Model):
    prescription = models.ForeignKey(Prescription, related_name='prescription_note', on_delete=models.CASCADE)
    diagnostic_body = models.TextField()
    updated_at = models.DateTimeField(auto_now=True)
    created_at = models.DateTimeField(auto_now_add=True)

class PrescriptionTherapyNotes(models.Model):
    prescription = models.ForeignKey(Prescription, related_name='therapy_note', on_delete=models.CASCADE)
    therapy_note = models.TextField()
    updated_at = models.DateTimeField(auto_now=True)
    created_at = models.DateTimeField(auto_now_add=True)

class DossierNoteHistory(models.Model):
    """
    patient dossier note history
    """
    doctor = models.ForeignKey('doctor.Doctor', on_delete=models.CASCADE, related_name='doctor_history_note')
    dossier = models.ForeignKey(Dossier, on_delete=models.CASCADE, related_name='dossier_note_history')
    dossier_history_note = models.TextField(blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

