from rest_framework import serializers
from django.utils.translation import gettext_lazy as _

from apps.referral.models import Referral, ReferralTherapyNotes
from apps.patient.models import Dossier, Patient
from apps.doctor.models import Doctor


class ReferralTherapyNotesSerializer(serializers.ModelSerializer):
    class Meta:
        model = ReferralTherapyNotes
        fields = '__all__'
        extra_kwargs = {
            'referral': {'read_only': True},
        }


class ReferralSerializer(serializers.ModelSerializer):
    referral_therapy_notes = ReferralTherapyNotesSerializer(many=True)
    from_doctor_name = serializers.SerializerMethodField()
    to_doctor_name = serializers.SerializerMethodField()

    class Meta:
        model = Referral
        fields = '__all__'
        extra_kwargs = {
            "is_active": {"read_only": True},
            'from_doctor': {"read_only": True},
        }

    def get_from_doctor_name(self, obj):
        return obj.from_doctor.name

    def get_to_doctor_name(self, obj):
        return obj.to_doctor.name

    def update(self, instance, validated_data):
        # pop referral therapeutic note
        referral_therapy_notes_data = validated_data.pop("referral_therapy_notes", [])

        # get and update data
        # instance.referral_date = validated_data.get('referral_date', instance.referral_date)
        instance.reason = validated_data.get('reason', instance.reason)
        instance.reason_detail = validated_data.get('reason_detail', instance.reason_detail)
        instance.patient = validated_data.get('patient', instance.patient)
        instance.to_doctor = validated_data.get('to_doctor', instance.to_doctor)
        instance.dossier = validated_data.get('dossier', instance.dossier)
        instance.save()

        # check referral therapeutic note, if exists
        if referral_therapy_notes_data:
            for note_data in referral_therapy_notes_data:
                note, created = ReferralTherapyNotes.objects.update_or_create(
                    referral=instance,
                    note=note_data.get('note'),
                    defaults={'note': note_data['note']}
                )

        return instance


class CreateUpdateReferralSerializer(serializers.ModelSerializer):
    referral_therapy_notes = ReferralTherapyNotesSerializer(many=True, required=False)

    class Meta:
        model = Referral
        fields = ["id", "dossier", "from_doctor", "to_doctor", "patient", "created_at", "reason",
                  "reason_detail", "referral_therapy_notes"]
        read_only_fields = ['from_doctor', 'patient']


    def create(self, validated_data):
        referral_therapy_notes = validated_data.pop("referral_therapy_notes", [])
        from_doctor = self.context['request'].user.doctor
        dossier = validated_data['dossier']

        referral, _ = Referral.objects.get_or_create(
            from_doctor=from_doctor, patient=dossier.patient, **validated_data)

        for therapeutic in referral_therapy_notes:
            ReferralTherapyNotes.objects.create(referral=referral, **therapeutic)

        return referral