from rest_framework import status
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView
from rest_framework.response import Response
from apps.doctor.permissions import IsDoctor
from apps.referral.models import Referral
from .serializers import CreateUpdateReferralSerializer, ReferralSerializer

class CreateReferralApiView(APIView):
    serializer_class = CreateUpdateReferralSerializer
    permission_classes = [IsDoctor]

    def post(self, request, *args, **kwargs):
        ser_data = self.serializer_class(data=request.data, context={'request': request})
        ser_data.is_valid(raise_exception=True)
        ser_data.save()
        return Response(ser_data.data, status=status.HTTP_201_CREATED)

class ReferralApiView(APIView):
    serializer_class = ReferralSerializer
    permission_classes = [IsAuthenticated]
    query = Referral.objects.select_related("patient", "from_doctor", "to_doctor", "dossier")\
        .filter(is_active=True)

    def get(self, request):
        referral_user_role = request.query_params.get('referral_user_role')
        if referral_user_role not in ("patient", "from_doctor", "to_doctor"):
            return Response({"detail": "Invalid referral_user_role. Must be 'patient', 'from_doctor', or 'to_doctor'."}, status=status.HTTP_200_OK)
        
        if referral_user_role == "from_doctor":
            referral = self.query.filter(from_doctor__user=request.user)
        elif referral_user_role == "to_doctor":
            referral = self.query.filter(to_doctor__user=request.user)
        elif referral_user_role == "patient":
            referral = self.query.filter(patient__user=request.user)
        
        serializer = self.serializer_class(referral, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)

class DetailsReferralApiView(APIView):
    """
    detail referral
    """
    serializer_class = ReferralSerializer
    query = Referral.objects.select_related("from_doctor", "to_doctor")
    permission_classes = [IsAuthenticated]

    def get(self, request, *args, **kwargs):
        referral = get_object_or_404(self.query, pk=kwargs['pk'], is_active=True)
        ser_data = self.serializer_class(referral)
        return Response(ser_data.data, status=status.HTTP_200_OK)