from django.db import models
from django.contrib.auth import get_user_model

User = get_user_model()

class AuthCodes(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE, unique=True)
    code_verifier = models.CharField(max_length=255)
    code_challenge = models.CharField(max_length=255)
    access_token = models.CharField(max_length=2000, blank=True, null=True)
    access_token_expires_at = models.DateTimeField(blank=True, null=True)
    last_callback_url = models.CharField(max_length=510, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return str(self.user)

class PrescriptionType(models.Model):
    code = models.CharField(max_length=8)
    desc = models.CharField(max_length=120, blank=True, null=True)

    def __str__(self):
        return f"{self.desc} - {self.code}"

class ServiceType(models.Model):
    code = models.CharField(max_length=8)
    desc = models.CharField(max_length=120, blank=True, null=True)
    status = models.CharField(max_length=8, blank=True, null=True)
    status_date = models.CharField(max_length=10, blank=True, null=True)
    cost_type = models.CharField(max_length=8, blank=True, null=True)
    presc_type_id = models.IntegerField(blank=True, null=True)
    head_expire_date = models.IntegerField(blank=True, null=True)

    def __str__(self):
        return f"{self.desc} - {self.code}"

class DrugInstruction(models.Model):
    code = models.CharField(max_length=20)
    summary = models.TextField(blank=True, null=True)
    latin = models.TextField(blank=True, null=True)
    concept = models.CharField(max_length=255)

    def __str__(self):
        return f"{self.concept} - {self.code}"

class ParTarefValue(models.Model):
    grp_code = models.CharField(max_length=8)
    grp_desc = models.CharField(max_length=120)
    grp_rem = models.CharField(max_length=8)
    status = models.CharField(max_length=8)
    status_date = models.CharField(max_length=10)

    def __str__(self):
        return f"{self.grp_desc} - {self.grp_code}"

class DrugAmount(models.Model):
    code = models.CharField(max_length=100)
    summary = models.TextField(blank=True, null=True)
    latin = models.CharField(max_length=100, blank=True, null=True)
    concept = models.CharField(max_length=255)
    visible = models.CharField(max_length=8, blank=True, null=True)

    def __str__(self):
        return f"{self.concept} - {self.code}"

class DrugUsage(models.Model):
    code = models.CharField(max_length=20)
    summary = models.TextField(blank=True, null=True)
    latin = models.TextField(blank=True, null=True)
    concept = models.CharField(max_length=255)
    visible = models.CharField(max_length=8, blank=True, null=True)
    drug_form_code = models.CharField(max_length=20, blank=True, null=True)

    def __str__(self):
        return f"{self.concept} - {self.code}"

class TreatmentPlan(models.Model):
    desc = models.CharField(max_length=120)
    code = models.CharField(max_length=20)

    def __str__(self):
        return f"{self.desc} - {self.code}"

class Illness(models.Model):
    desc = models.CharField(max_length=255)

    def __str__(self):
        return self.desc

class Complaint(models.Model):
    icd_id = models.CharField(max_length=20)
    icd_code = models.CharField(max_length=20)
    icd_name = models.TextField(blank=True, null=True)

    def __str__(self):
        return f"{self.icd_name} - {self.icd_code}"

class ICD10Code(models.Model):
    code = models.IntegerField()
    display_name = models.CharField(max_length=255)
    english_name = models.CharField(max_length=255)
    terminology = models.CharField(max_length=50)
    status = models.CharField(max_length=8)

    def __str__(self):
        return f"{self.display_name} - {self.code}"

class MedicalSpecialty(models.Model):
    code = models.CharField(max_length=20)
    desc = models.CharField(max_length=255)
    grp = models.CharField(max_length=20)
    doc_comment = models.TextField(blank=True, null=True)
    status = models.CharField(max_length=8)
    status_date = models.CharField(max_length=10)
    type_spec = models.CharField(max_length=20, blank=True, null=True)
    max_normal = models.IntegerField(blank=True, null=True)
    max_special = models.IntegerField(blank=True, null=True)
    l_status = models.CharField(max_length=8)

    def __str__(self):
        return f"{self.desc} - {self.code}"

class Service(models.Model):
    service_type = models.ForeignKey(ServiceType, on_delete=models.CASCADE, related_name='services')
    srv_id = models.CharField(max_length=20, blank=True, null=True)
    srv_type = models.JSONField(default=dict)
    srv_code = models.CharField(max_length=20, blank=True, null=True)
    name = models.TextField(blank=True, null=True)
    name2 = models.CharField(max_length=500, blank=True, null=True)
    bim_sw = models.CharField(max_length=500, blank=True, null=True)
    sex = models.CharField(max_length=8, blank=True, null=True)
    price = models.CharField(max_length=120, blank=True, null=True)
    price_date = models.CharField(max_length=15, blank=True, null=True)
    dose_code = models.CharField(max_length=500, blank=True, null=True)
    form_code = models.CharField(max_length=500, blank=True, null=True)
    par_taref_group = models.JSONField(default=dict, blank=True, null=True)
    status = models.CharField(max_length=500, blank=True, null=True)
    status_date = models.CharField(max_length=500, blank=True, null=True)
    bg_type = models.CharField(max_length=500, blank=True, null=True)
    gsrv_code = models.CharField(max_length=500, blank=True, null=True)
    agreement_flag = models.CharField(max_length=500, blank=True, null=True)
    is_deleted = models.CharField(max_length=500, blank=True, null=True)
    visible = models.CharField(max_length=500, blank=True, null=True)
    dental_service_type = models.CharField(max_length=500, blank=True, null=True)
    ws_srv_code = models.CharField(max_length=500, blank=True, null=True)
    hos_presc_type = models.CharField(max_length=500, blank=True, null=True)
    srv_rule = models.CharField(max_length=500, blank=True, null=True)
    count_is_restricted = models.CharField(max_length=500, blank=True, null=True)
    terminology = models.CharField(max_length=500, blank=True, null=True)
    srv_code_complete = models.CharField(max_length=500, blank=True, null=True)

    def __str__(self):
        return f"{self.name} - {self.ws_srv_code}"

class NoteDetailsReferral(models.Model):
    referral_hijri_date = models.CharField(max_length=10)
    patient_national_code = models.CharField(max_length=20)
    doc_spec_referred = models.ForeignKey(MedicalSpecialty, on_delete=models.RESTRICT)
    message = models.TextField()
    icd10s = models.ManyToManyField(ICD10Code)
    complaints = models.ManyToManyField(Complaint)
    quantity = models.IntegerField()

class NoteDetailEprsc(models.Model):
    # Service, DrugAmount, DrugInstruction, Illness, TreatmentPlan should be related to existing models
    srv = models.ForeignKey(Service, on_delete=models.RESTRICT)
    srv_qty = models.IntegerField()
    times_a_day = models.ForeignKey(DrugAmount, on_delete=models.SET_NULL, null=True, blank=True)
    dose = models.CharField(max_length=64, null=True, blank=True)
    repeat = models.CharField(max_length=32, null=True, blank=True)
    date_do = models.CharField(max_length=10, null=True, blank=True)
    drug_instruction = models.ForeignKey(DrugInstruction, on_delete=models.SET_NULL, null=True, blank=True)
    illness = models.ForeignKey(Illness, on_delete=models.SET_NULL, null=True, blank=True)
    plan = models.ForeignKey(TreatmentPlan, on_delete=models.SET_NULL, null=True, blank=True)
    reference_status = models.IntegerField(null=True, blank=True)
    is_dental_service = models.CharField(max_length=8, null=True, blank=True)
    tooth_id = models.CharField(max_length=16, null=True, blank=True)
    note_details_eprsc_id = models.IntegerField(null=True, blank=True)

class Prescription(models.Model):
    patient_user = models.ForeignKey(User, on_delete=models.SET_NULL, blank=True, null=True, related_name='tamin_prescriptions')
    doctor_user = models.ForeignKey(User, on_delete=models.SET_NULL, blank=True, null=True, related_name='doctor_tamin_prescriptions')
    head_id = models.CharField(max_length=120, unique=True)
    patient = models.CharField(max_length=20)
    presc_date = models.CharField(max_length=10)
    expire_date = models.CharField(max_length=10)
    doc_id = models.CharField(max_length=20)
    doc_national_code = models.CharField(max_length=20)
    doc_mobile_no = models.CharField(max_length=20)
    presc_type = models.ForeignKey(PrescriptionType, on_delete=models.RESTRICT)
    comments = models.TextField(null=True, blank=True)
    note_detail_eprscs = models.ManyToManyField(NoteDetailEprsc, blank=True)
    note_details_referral_list = models.ManyToManyField(NoteDetailsReferral, blank=True)
    referral_feedback_id = models.IntegerField(null=True, blank=True)
    client_id = models.CharField(max_length=64, null=True, blank=True)
    siam_id = models.CharField(max_length=64, null=True, blank=True)
    creator_type = models.CharField(max_length=32, null=True, blank=True)

    def __str__(self):
        return f"Prescription for {self.patient} - {self.presc_date}"