from .tamin_sdk.models import Prescription as SDKPrescription, NoteDetailEprsc as SDKNoteDetailEprsc, NoteDetailsReferral as SDKNoteDetailsReferral, Service as SDKService, ServiceType as SDKServiceType, DrugInstruction as SDKDrugInstruction, TbDrugAmount, PrescriptionType as SDKPrescriptionType, PrescriptionTypeId, ServiceTypeId, Special as SDKSpecial, Icd10 as SDKIcd10, Complaint as SDKComplaint, ParTarefGroup as SDKPartarefGroup, ParTarefGroupCode
from typing import List, Optional

def build_sdk_prescription(validated_data):
    """
    Convert validated_data from PrescriptionSerializer to SDKPrescription instance.
    Handles mapping of local DB IDs to SDK codes/objects.
    """
    # Map prescription type
    presc_type_obj = validated_data.get('presc_type')
    sdk_presc_type = SDKPrescriptionType(prescTypeId=presc_type_obj.code)

    # Map note_detail_eprscs
    sdk_note_detail_eprscs = []
    for eprsc in validated_data.get('note_detail_eprscs', []):
        srv = eprsc.get('srv')
        srv_type_code = srv.service_type.code
        sdk_service = SDKService(
            srvCode=srv.ws_srv_code,
            srvType=SDKServiceType(srvType=ServiceTypeId(srv_type_code)),
            parTarefGrp=SDKPartarefGroup(parGrpCode=ParTarefGroupCode(srv.par_taref_group['parGrpCode'])) if srv.par_taref_group else None,
        )
        sdk_eprsc = SDKNoteDetailEprsc(
            srvId=sdk_service,
            srvQty=eprsc.get('srv_qty'),
            timesAday=TbDrugAmount(drugAmntId=eprsc['times_a_day'].id) if eprsc.get('times_a_day') else None,
            dose=eprsc.get('dose'),
            repeat=eprsc.get('repeat'),
            dateDo=eprsc.get('date_do'),
            drugInstruction=SDKDrugInstruction(drugInstId=eprsc['drug_instruction'].id) if eprsc.get('drug_instruction') else None,
            illnessId=str(eprsc['illness'].id) if eprsc.get('illness') else None,
            planId=str(eprsc['plan'].code) if eprsc.get('plan') else None,
            referenceStatus=eprsc.get('reference_status'),
            isDentalService=eprsc.get('is_dental_service'),
            toothId=eprsc.get('tooth_id'),
        )
        sdk_note_detail_eprscs.append(sdk_eprsc)

    # Map note_details_referral_list
    sdk_note_details_referral_list = []
    for referral in validated_data.get('note_details_referral_list', []):
        doc_spec = referral.get('doc_spec_referred')
        icd10s = [SDKIcd10(icdId=icd.code) for icd in referral.get('icd10s', [])]
        complaints = [SDKComplaint(id=complaint.icd_code) for complaint in referral.get('complaints', [])]
        sdk_referral = SDKNoteDetailsReferral(
            id=referral.get('id'),
            referralHijriDate=referral.get('referral_hijri_date'),
            patientNationalCode=referral.get('patient_national_code'),
            docSpecReferred=SDKSpecial(specCode=doc_spec.code),
            message=referral.get('message'),
            icd10s=icd10s,
            complaints=complaints,
            quantity=referral.get('quantity'),
        )
        sdk_note_details_referral_list.append(sdk_referral)

    sdk_presc = SDKPrescription(
        patient=validated_data.get('patient'),
        prescDate=validated_data.get('presc_date'),
        expireDate=validated_data.get('expire_date'),
        docId=validated_data.get('doc_id'),
        docNationalCode=validated_data.get('doc_national_code'),
        docMobileNo=validated_data.get('doc_mobile_no'),
        prescType=sdk_presc_type,
        comments=validated_data.get('comments'),
        noteDetailEprscs=sdk_note_detail_eprscs,
        noteDetailsReferralList=sdk_note_details_referral_list or None,
        referralFeedbackId=validated_data.get('referral_feedback_id'),
    )
    return sdk_presc
