import base64
import hashlib
import secrets
from typing import Optional, Dict, Any
from urllib.parse import urlencode

import requests
from pydantic import BaseModel

from .models import Prescription

class TaminClient:
    """Main client for interacting with Tamin Insurance API."""
    
    def __init__(
        self,
        client_id: str,
        redirect_uri: str,
        test_environment: bool = False,
    ):
        """
        Initialize the Tamin client.
        
        Args:
            client_id: The client ID provided by the insurance company
            redirect_uri: The URI to redirect to after authentication
            test_environment: Bool indicating if the test environment should be used
        """
        self.client_id = client_id
        self.redirect_uri = redirect_uri
        self.test_environment = test_environment
        self._access_token: Optional[str] = None
        
        # Set base URLs based on environment
        if test_environment:
            self.auth_base_url = "http://account-test.tamin.ir:9090/auth/server"
            self.api_base_url = "https://ep-test.tamin.ir/api"
        else:
            self.auth_base_url = "http://account.tamin.ir/auth/server"
            self.api_base_url = "https://soa.tamin.ir/interface/epresc"
    
    def generate_code_verifier(self) -> str:
        """Generate a code verifier for PKCE authentication."""
        return secrets.token_urlsafe(120)
    
    def generate_code_challenge(self, code_verifier: str) -> str:
        """Generate a code challenge from a code verifier."""
        sha256_hash = hashlib.sha256(code_verifier.encode()).digest()
        return base64.urlsafe_b64encode(sha256_hash).decode().rstrip("=")
    
    def get_authentication_url(self, code_challenge: str) -> str:
        """
        Get the authentication URL for the user to log in.
        
        Args:
            code_challenge: The code challenge generated from the code verifier
            
        Returns:
            The URL to redirect the user to for authentication
        """
        params = {
            "redirect_uri": self.redirect_uri,
            "code_challenge": code_challenge,
            "client_id": self.client_id,
            "response_type": "code",
            "code_challenge_method": "S256"
        }
        return f"{self.auth_base_url}/authorize?{urlencode(params)}"
    
    def get_access_token(self, code: str, code_verifier: str) -> str:
        """
        Exchange the authorization code for an access token.
        
        Args:
            code: The authorization code received from the redirect
            code_verifier: The code verifier used to generate the code challenge
            
        Returns:
            The access token
        """
        data = {
            "redirect_uri": self.redirect_uri,
            "grant_type": "authorization_code",
            "client_id": self.client_id,
            "code": code,
            "code_verifier": code_verifier
        }
        
        response = requests.post(
            f"{self.auth_base_url}/token",
            data=data
        )
        response.raise_for_status()
        
        token_data = response.json()
        self._access_token = token_data["access_token"]
        return self._access_token
    
    def register_prescription(self, prescription: Prescription) -> Dict[str, Any]:
        """
        Register a new prescription.
        
        Args:
            prescription: The prescription data to register
            
        Returns:
            The response from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        url = f"{self.api_base_url}/SendEpresc/v2" if not self.test_environment else f"{self.api_base_url}/v2/SendEpresc/"
        headers = {"Authorization": f"Bearer {self._access_token}"}
        payload = prescription.model_dump(exclude_none=True)
        if self.test_environment:
            payload['clientId'] = self.client_id
        print(payload)
        response = requests.post(
            url,
            json=payload,
            headers=headers
        )
        response.raise_for_status()
        return response.json()['data']
    
    def check_patient_credibility(
        self,
        siam_id: str,
        doc_id: str,
        patient_id: str,
        request_by: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Check if a patient can use insurance services.
        
        Args:
            siam_id: The SIAM identifier for the clinic
            doc_id: Doctor's medical system ID
            patient_id: Patient's national ID
            request_by: Doctor's national code (only in test environment)
            
        Returns:
            The response from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/patients/deserve-info/{request_by}/{siam_id}/{doc_id}/{patient_id}"
        else:
            url = f"{self.api_base_url}/patient/v2/deserve-info/{siam_id}/{doc_id}/{patient_id}"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()
    
    def fetch_prescription(
        self,
        header_id: str,
        doc_id: str,
        doc_national_code: str
    ) -> Dict[str, Any]:
        """
        Fetch prescription details.
        
        Args:
            header_id: Prescription's unique identifier
            doc_id: Doctor's medical system ID
            doc_national_code: Doctor's national code
            
        Returns:
            The prescription details
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ep/{header_id}/{doc_national_code}/{doc_id}/detail"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/{header_id}/{doc_id}"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()

    def remove_prescription(
        self,
        header_id: str,
        doc_id: str,
        doc_national_code: str,
        otp_code: str
    ) -> Dict[str, Any]:
        """
        Remove a prescription.
        
        Args:
            header_id: Prescription's unique identifier
            doc_id: Doctor's medical system ID
            doc_national_code: Doctor's national code
            otp_code: OTP code sent to doctor's mobile phone
            
        Returns:
            The response from the API
            
        Note:
            Prescriptions can only be deleted during the same day they are registered.
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ep/{header_id}/{doc_national_code}/{doc_id}/{otp_code}"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/remove/{header_id}/{doc_id}/{otp_code}"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        
        response = requests.post(url, headers=headers)
        response.raise_for_status()
        return response.json()

    def update_prescription(
        self,
        header_id: str,
        doc_id: str,
        doc_national_code: str,
        otp_code: str,
        prescription: Prescription
    ) -> Dict[str, Any]:
        """
        Update a prescription.
        
        Args:
            header_id: Prescription's unique identifier
            doc_id: Doctor's medical system ID
            doc_national_code: Doctor's national code
            otp_code: OTP code sent to doctor's mobile phone
            prescription: Updated prescription data
            
        Returns:
            The response from the API
            
        Note:
            - Prescriptions can only be updated before pharmacy or paraclinic take any actions
            - Prescriptions can only be updated using the same clientId they were registered with
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ep/update/{header_id}/{doc_national_code}/{doc_id}/{otp_code}"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/edit/{header_id}/{doc_id}/{otp_code}"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        
        response = requests.post(
            url,
            json=prescription.model_dump(exclude_none=True),
            headers=headers
        )
        response.raise_for_status()
        return response.json()

    def get_prescription_types(self) -> Dict[str, Any]:
        """
        Get the list of e-prescription types.
        
        Returns:
            The list of prescription types from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-prescription-type"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/prescription-type"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_service_types(self) -> Dict[str, Any]:
        """
        Get the list of service types.
        
        Returns:
            The list of service types from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-service-type"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/service-type"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_services(self, service_type: str) -> Dict[str, Any]:
        """
        Get the list of medication, paraclinic and service codings.
        
        Args:
            service_type: The type of service to get codings for
            
        Returns:
            The list of services from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-services?serviceType={service_type}"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/srvices?service-type={service_type}"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_par_taref_values(self) -> Dict[str, Any]:
        """
        Get the list of subtypes for lab prescriptions (par-taref values).
        
        Returns:
            The list of par-taref values from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-par-taref"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/par-taref"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_drug_amounts(self) -> Dict[str, Any]:
        """
        Get the list of drug amount values.
        
        Returns:
            The list of drug amounts from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-drug-amount"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/drug-amount"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_drug_usages(self) -> Dict[str, Any]:
        """
        Get the list of drug usage values.
        
        Returns:
            The list of drug usages from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-drug-usage"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/drug-usage"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_drug_instructions(self) -> Dict[str, Any]:
        """
        Get the list of drug instruction values.
        
        Returns:
            The list of drug instructions from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-drug-instruction"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/drug-instruction"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_treatment_plans(self) -> Dict[str, Any]:
        """
        Get the list of treatment plan values.
        
        Returns:
            The list of treatment plans from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-ph-plan"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/ph-plan"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_illnesses(self) -> Dict[str, Any]:
        """
        Get the list of illness values.
        
        Returns:
            The list of illnesses from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/v2/ws-ph-illness"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/ph-illness"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']

    def get_complaints(self) -> Dict[str, Any]:
        """
        Get the list of complaint values.
        
        Returns:
            The list of complaints from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/icd10/getAll"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/complaint"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']['list']

    def get_icd10_codes(self) -> Dict[str, Any]:
        """
        Get the list of ICD10 diagnosis codes.
        
        Returns:
            The list of ICD10 codes from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/complaint"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/icd10/getAll"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']['list']

    def get_medical_specialties(self) -> Dict[str, Any]:
        """
        Get the list of medical specialties.
        
        Returns:
            The list of medical specialties from the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        if self.test_environment:
            url = f"{self.api_base_url}/specials/v2/getAll"
        else:
            url = f"{self.api_base_url}/SendEpresc/v2/special/getAll"
        
        headers = {"Authorization": f"Bearer {self._access_token}"}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        return response.json()['data']['list']
    
    def get_basic_data(self) -> Dict[str, Any]:
        """
        Returns:
            A dictionary of all basic services' outputs ftom the API
        """
        if not self._access_token:
            raise ValueError("Not authenticated. Call get_access_token first.")
        
        basic_data = {
            'prescription_types': self.get_prescription_types(),
            'service_types': self.get_service_types(),
            'drug_instructions': self.get_drug_instructions(),
            'par_taref_values': self.get_par_taref_values(),
            'drug_amounts': self.get_drug_amounts(),
            'drug_usages': self.get_drug_usages(),
            'treatment_plans': self.get_treatment_plans(),
            'illnesses': self.get_illnesses(),
            'complaints': self.get_complaints(),
            'icd10_codes': self.get_icd10_codes(),
            'medical_specialties': self.get_medical_specialties()
        }
        basic_data['service_codes'] = {}
        for service_type in basic_data['service_types']:
            basic_data['service_codes'][service_type['srvType']] = self.get_services(service_type['srvType'])
        return basic_data
    