from enum import Enum
from typing import List, Optional
from pydantic import BaseModel, Field

class PrescriptionTypeId(int, Enum):
    MEDICATION = 1
    PARACLINIC = 2
    VISIT = 3
    SERVICES = 5
    REFERRAL_SERVICES = 6
    DOCTOR_REFERRAL = 7
    DOCTOR_REFERRAL_FEEDBACK = 8

class ServiceTypeId(str, Enum):
    MEDICATION = "01"
    LAB = "02"
    RADIOLOGY = "03"
    SONOGRAPHY = "04"
    CT_SCAN = "05"
    MRI = "06"
    NUCLEAR_MEDICINE = "07"
    RADIOTHERAPY = "08"
    AUDIOMETRY = "09"
    SPEECH_THERAPY = "10"
    ANGIOGRAPHY = "11"
    ADDITIONAL_SERVICES = "12"
    PHYSICAL_THERAPY = "13"
    BONE_DENSITY_TEST = "14"
    DIALYSIS = "15"
    VISIT = "16"
    DENTAL = "17"
    NURSARY = "21"

class ParTarefGroupCode(str, Enum):
    PATHOLOGY = "001"
    GENETIC = "002"
    GLOBAL_GENETIC = "003"
    UNKNOWN = "000"
    MEDICAL_DIAGNOSIS = "004"

class PrescriptionType(BaseModel):
    prescTypeId: PrescriptionTypeId

class ServiceType(BaseModel):
    srvType: ServiceTypeId

class ParTarefGroup(BaseModel):
    parGrpCode: ParTarefGroupCode

class Service(BaseModel):
    srvCode: str
    srvType: ServiceType
    parTarefGrp: Optional[ParTarefGroup] = None
    terminology: Optional[str] = None

class TbDrugAmount(BaseModel):
    drugAmntId: int

class DrugInstruction(BaseModel):
    drugInstId: int

class NoteDetailEprsc(BaseModel):
    srvId: Service
    srvQty: int = Field(description="Quantity of the service/medication to be provided")
    timesAday: Optional[TbDrugAmount] = Field(None, description="Number of times per day the medication should be taken")
    dose: Optional[str] = Field(None, description="Dosage of medication (e.g., '1 tablet', '5ml')")
    repeat: Optional[str] = Field(None, description="Number of times the prescription can be refilled")
    dateDo: Optional[str] = Field(None, description="Date when the service should be performed (format: YYYYMMDD)")
    drugInstruction: Optional[DrugInstruction] = Field(None, description="Special instructions for drug usage")
    illnessId: Optional[str] = Field(None, description="Identifier for the illness being treated")
    planId: Optional[str] = Field(None, description="Treatment plan identifier")
    referenceStatus: Optional[int] = Field(None, description="Status code for referrals")
    isDentalService: Optional[str] = Field(None, description="Flag indicating if this is a dental service")
    toothId: Optional[str] = Field(None, description="Identifier for the specific tooth (for dental services)")
    noteDetailsEprscId: Optional[int] = Field(None, description="Unique identifier for this prescription detail")

class Special(BaseModel):
    specCode: str = Field(description="Medical specialty code for the referred doctor")

class Icd10(BaseModel):
    icdId: str = Field(description="ICD-10 diagnosis code")

class Complaint(BaseModel):
    id: str = Field(description="Unique identifier for the patient's complaint")

class NoteDetailsReferral(BaseModel):
    id: Optional[int] = Field(None, description="Unique identifier for the referral")
    referralHijriDate: str = Field(description="Date of referral in Hijri format (YYYYMMDD)")
    patientNationalCode: str = Field(description="Patient's national identification code")
    docSpecReferred: Special = Field(description="Specialty of the doctor being referred to")
    message: str = Field(description="Referral notes or instructions")
    icd10s: List[Icd10] = Field(description="List of ICD-10 diagnosis codes")
    complaints: List[Complaint] = Field(description="List of patient complaints")
    quantity: int = Field(description="Number of visits allowed with this referral")

class Prescription(BaseModel):
    patient: str = Field(description="Patient's national identification code")
    prescDate: str = Field(description="Prescription issue date in Hijri format (YYYYMMDD)")
    expireDate: str = Field(description="Prescription expiration date in Hijri format (YYYYMMDD)")
    docId: str = Field(description="Doctor's medical system ID")
    docNationalCode: str = Field(description="Doctor's national identification code")
    docMobileNo: str = Field(description="Doctor's mobile phone number")
    prescType: PrescriptionType = Field(description="Type of prescription (medication, paraclinic, visit, etc.)")
    comments: Optional[str] = Field(None, description="Additional notes or instructions for the prescription")
    noteDetailEprscs: List[NoteDetailEprsc] = Field(description="List of prescribed medications or services")
    noteDetailsReferralList: Optional[List[NoteDetailsReferral]] = Field(None, description="List of referrals (for referral prescriptions)")
    referralFeedbackId: Optional[int] = Field(None, description="ID of the original referral (for feedback prescriptions)")
    clientId: Optional[str] = Field(None, description="Client system identifier")
    siamId: Optional[str] = Field(None, description="SIAM system identifier for the clinic")
    creatorType: Optional[str] = Field(None, description="Type of user creating the prescription")