from datetime import timedelta
import jdatetime
from django.shortcuts import redirect
from django.utils import timezone
from django.db.models import Q
from rest_framework import status,  permissions, generics
from rest_framework.views import APIView
from rest_framework.response import Response
from .serializers import (PrescriptionTypeSerializer, ServiceTypeSerializer,
                          DrugInstructionSerializer, ParTarefValueSerializer,
                          DrugAmountSerializer, DrugUsageSerializer,
                          TreatmentPlanSerializer, IllnessSerializer, ComplaintSerializer,
                          ICD10CodeSerializer, MedicalSpecialtySerializer, ServiceSerializer,
                          PrescriptionSerializer)
from .models import (AuthCodes, PrescriptionType,
                     ServiceType, DrugInstruction, ParTarefValue, DrugAmount,
                     DrugUsage, TreatmentPlan, Illness, Complaint, ICD10Code, MedicalSpecialty, Service,
                     Prescription)
from .tamin_sdk.models import Prescription as SDKPrescription, NoteDetailEprsc as SDKNoteDetailEprsc, NoteDetailsReferral as SDKNoteDetailsReferral, Special as SDKSpecial, Icd10 as SDKIcd10, Complaint as SDKComplaint
from .tamin_sdk.client import TaminClient
from .config import TAMIN_CLIENT_ID, TAMIN_USE_TEST_ENV, TAMIN_REDIRECT_URL
from .sdk_utils import build_sdk_prescription
from .paginations import CustomPageNumberPagination

class AuthenticateURLView(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def post(self, request, *args, **kwargs):
        redirect_url = request.data.get('redirect_url', None)
        callback_url = request.data.get('callback_url', None)
        client = TaminClient(
            client_id=TAMIN_CLIENT_ID,
            test_environment=TAMIN_USE_TEST_ENV,
            redirect_uri=f"{redirect_url}?u_id={request.user.id}" if redirect_url is not None else f"{TAMIN_REDIRECT_URL}?u_id={request.user.id}"
        )
        code_verifier = client.generate_code_verifier()
        code_challenge = client.generate_code_challenge(code_verifier)
        auth_code_pair, _ = AuthCodes.objects.update_or_create(
            user=request.user,
            defaults={
                'code_verifier': code_verifier,
                'code_challenge': code_challenge,
                'last_callback_url': callback_url
            }
        )
        auth_url = client.get_authentication_url(code_challenge)
        return Response({
            "auth_url": auth_url
        })

class AuthenticateAPIView(APIView):
    permission_classes = [permissions.AllowAny]

    def get(self, request, *args, **kwargs):
        user_id = request.query_params.get('u_id', None)
        if not user_id:
            return Response({
                "error": "User ID is required."
            }, status=status.HTTP_400_BAD_REQUEST)
        code = request.query_params.get('code', None)
        if not code:
            return Response({
                "error": "Authorization code is required."
            }, status=status.HTTP_400_BAD_REQUEST)
        auth_codes = AuthCodes.objects.filter(user_id=user_id).first()
        if auth_codes is None:
            return Response({
                "error": "No authorization codes requested for user."
            }, status=status.HTTP_404_NOT_FOUND)
        client = TaminClient(
            client_id=TAMIN_CLIENT_ID,
            test_environment=TAMIN_USE_TEST_ENV,
            redirect_uri=f"{TAMIN_REDIRECT_URL}?u_id={user_id}"
        )
        try:
            access_token = client.get_access_token(
                code=code,
                code_verifier=auth_codes.code_verifier
            )
        except Exception as e:
            print(f"Error retrieving access token: {e}")
            return Response({
                "error": "Failed to retrieve access token."
            }, status=status.HTTP_400_BAD_REQUEST)
        auth_codes.access_token = access_token
        auth_codes.access_token_expires_at = timezone.now() + timedelta(seconds=3480)
        auth_codes.save()
        if auth_codes.last_callback_url:
            return redirect(auth_codes.last_callback_url)
        return Response({}, status=status.HTTP_200_OK)

class CheckUserTaminToken(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def get(self, request, *args, **kwargs):
        auth_codes_record = AuthCodes.objects.filter(user=request.user).first()
        if not auth_codes_record or not auth_codes_record.access_token or auth_codes_record.access_token_expires_at < timezone.now():
            return Response({"authenticated": False}, status=status.HTTP_200_OK)
        return Response({"authenticated": True})

class CheckPatientCredability(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def get(self, request, patient_id, *args, **kwargs):
        try:
            doctor = request.user.doctor
        except:
            return Response(status=status.HTTP_403_FORBIDDEN)
        auth_codes_record = AuthCodes.objects.filter(user=request.user).first()
        if not auth_codes_record or not auth_codes_record.access_token or auth_codes_record.access_token_expires_at < timezone.now():
            return Response({"error": "User should complete tamin authentication first."}, status=status.HTTP_401_UNAUTHORIZED)
        client = TaminClient(
            client_id=TAMIN_CLIENT_ID,
            redirect_uri=f"{TAMIN_REDIRECT_URL}?u_id={request.user.id}",
            test_environment=TAMIN_USE_TEST_ENV
        )
        client._access_token = auth_codes_record.access_token
        try:
            res = client.check_patient_credibility(
                siam_id=doctor.medical_system_code,
                doc_id=doctor.medical_system_code,
                patient_id=patient_id,
                request_by=doctor.nation_code
            )
        except Exception as e:
            print(f"Error occured during check paitent credability. {e}")
            return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)
        return Response(res, status=status.HTTP_200_OK)

# Basic data retrieval views
class PrescriptionTypeListView(generics.ListAPIView):
    serializer_class = PrescriptionTypeSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = PrescriptionType.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(desc__icontains=keyword)
        return queryset

class ServiceTypeListView(generics.ListAPIView):
    serializer_class = ServiceTypeSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = ServiceType.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(desc__icontains=keyword)
        return queryset

class DrugInstructionListView(generics.ListAPIView):
    serializer_class = DrugInstructionSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = DrugInstruction.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(code__icontains=keyword) | Q(concept__icontains=keyword) \
                    | Q(summary__icontains=keyword) | Q(latin__icontains=keyword)
                )
        return queryset

class ParTarefValueListView(generics.ListAPIView):
    serializer_class = ParTarefValueSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = ParTarefValue.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(grp_desc__icontains=keyword) | Q(grp_code__icontains=keyword) \
                    | Q(grp_rem__icontains=keyword)
                )
        return queryset

class DrugAmountListView(generics.ListAPIView):
    serializer_class = DrugAmountSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = DrugAmount.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(code__icontains=keyword) | Q(concept__icontains=keyword) \
                    | Q(summary__icontains=keyword) | Q(latin__icontains=keyword)
                )
        return queryset

class DrugUsageListView(generics.ListAPIView):
    serializer_class = DrugUsageSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = DrugUsage.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(code__icontains=keyword) | Q(concept__icontains=keyword) \
                    | Q(summary__icontains=keyword) | Q(latin__icontains=keyword)
                )
        return queryset

class TreatmentPlanListView(generics.ListAPIView):
    serializer_class = TreatmentPlanSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = TreatmentPlan.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(desc__icontains=keyword)
        return queryset

class IllnessListView(generics.ListAPIView):
    serializer_class = IllnessSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = Illness.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(desc__icontains=keyword)
        return queryset

class ComplaintListView(generics.ListAPIView):
    serializer_class = ComplaintSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = Complaint.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(icd_id__icontains=keyword) | Q(icd_code__icontains=keyword) | Q(icd_name__icontains=keyword)
            )
        return queryset

class ICD10CodeListView(generics.ListAPIView):
    serializer_class = ICD10CodeSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = ICD10Code.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(code__icontains=keyword) | Q(display_name__icontains=keyword) | Q(english_name__icontains=keyword) | Q(terminology__icontains=keyword)
            )
        return queryset

class MedicalSpecialtyListView(generics.ListAPIView):
    serializer_class = MedicalSpecialtySerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        queryset = MedicalSpecialty.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(code__icontains=keyword) | Q(desc__icontains=keyword) | Q(grp__icontains=keyword)
            )
        return queryset

class ServiceListView(generics.ListAPIView):
    serializer_class = ServiceSerializer
    permission_classes = [permissions.IsAuthenticated]
    pagination_class = CustomPageNumberPagination

    def get_queryset(self):
        service_type_id = self.kwargs.get('service_type_id', None)
        if service_type_id:
            queryset = Service.objects.filter(service_type_id=service_type_id)
        else:
            queryset = Service.objects.all()
        keyword = self.request.query_params.get('keyword', None)
        if keyword:
            queryset = queryset.filter(
                Q(srv_code__icontains=keyword) | Q(name__icontains=keyword) | Q(name2__icontains=keyword) | Q(terminology__icontains=keyword)
            )
        return queryset

class PrescriptionCreateView(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def post(self, request, *args, **kwargs):
        if not request.user.doctor:
            return Response({"error": "User is not a doctor."}, status=status.HTTP_403_FORBIDDEN)
        doctor = request.user.doctor
        auth_codes_record = AuthCodes.objects.filter(user=request.user).first()
        if not auth_codes_record or not auth_codes_record.access_token or auth_codes_record.access_token_expires_at < timezone.now():
            return Response({"error": "User should complete tamin authentication first."}, status=status.HTTP_401_UNAUTHORIZED)
        doc_id = doctor.medical_system_code
        if not doc_id:
            return Response({"error": "Doctor's medical system code is required."}, status=status.HTTP_400_BAD_REQUEST)
        doc_national_code = doctor.nation_code
        if not doc_national_code:
            return Response({"error": "Doctor's national code is required."}, status=status.HTTP_400_BAD_REQUEST)
        doc_mobile_no = request.user.mobile_phone

        # Merge/override fields in request.data
        data = request.data.copy()
        data['doc_id'] = doc_id
        data['doc_national_code'] = doc_national_code
        data['doc_mobile_no'] = doc_mobile_no
        data['presc_date'] = jdatetime.date.today().strftime('%Y%m%d')

        serializer = PrescriptionSerializer(data=data)
        if not serializer.is_valid():
            return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

        # Convert validated data to tamin_sdk.models.Prescription using utility
        try:
            sdk_presc = build_sdk_prescription(serializer.validated_data)
        except Exception as e:
            return Response({"sdk_validation_error": str(e)}, status=status.HTTP_400_BAD_REQUEST)

        # Register prescription with TaminClient
        client = TaminClient(
            client_id=TAMIN_CLIENT_ID,
            redirect_uri=f"{TAMIN_REDIRECT_URL}?u_id={request.user.id}",
            test_environment=TAMIN_USE_TEST_ENV
        )
        client._access_token = auth_codes_record.access_token
        try:
            api_response = client.register_prescription(sdk_presc)
        except Exception as e:
            return Response({"tamin_api_error": str(e)}, status=status.HTTP_400_BAD_REQUEST)

        # Use head field from API response for head_id
        head_id = api_response.get('result', {}).get('head_EPRSC_ID')
        if not head_id:
            print(api_response)
            return Response({"tamin_api_error": api_response}, status=status.HTTP_400_BAD_REQUEST)
        instance = serializer.save(head_id=head_id)
        return Response(PrescriptionSerializer(instance).data, status=status.HTTP_201_CREATED)