import random
from datetime import timedelta
from django.db import models
from django.utils import timezone
from django.contrib.auth.models import AbstractUser, UserManager as BaseUserManager
from django.utils.translation import gettext_lazy as _
from .validators import MobilePhoneValidator


class UserManager(BaseUserManager):
    def create_user(self, mobile_phone, password=None, **extra_fields):
        if not mobile_phone:
            raise ValueError('mobile_phone field must be set')
        return super().create_user(username=mobile_phone, mobile_phone=mobile_phone,
                                   password=password, **extra_fields)

    def create_superuser(self, mobile_phone, password, **extra_fields):
        return super().create_superuser(username=mobile_phone, mobile_phone=mobile_phone,
                                        password=password, **extra_fields)


class User(AbstractUser):
    """
    Custom user model that extends Django's AbstractUser.
    Uses mobile_phone as username field.
    """
    mobile_phone = models.CharField(_("mobile_phone"), max_length=11,
                                    unique=True, validators=[MobilePhoneValidator])
    otp = models.CharField(_("otp"), max_length=6, blank=True, null=True)
    otp_created_at = models.DateTimeField(_("otp created at"), blank=True, null=True)
    
    USERNAME_FIELD = 'mobile_phone'
    REQUIRED_FIELDS = []

    objects = UserManager()

    def is_first_time_login(self):
        """
        Check if the is logging in for the first time.
        """
        return self.last_login is None
    
    def is_doctor(self):
        """
        Check if the user is registered as a doctor.
        """
        return hasattr(self, 'doctor')
    
    def is_patient(self):
        """
        Check if the user is registered as a patient.
        """
        return hasattr(self, 'patient')
    
    def get_role_name(self):
        """
        Get's user's name.
        *Warning* This method will be deprecated in the future.
        It is only implemented for backward compatibility.
        """
        # TODO: Identify where in code this method is used, change it and then remove this method.
        return self.get_full_name()
    
    def is_otp_expired(self):
        return self.otp and (timezone.now() > self.otp_created_at + timedelta(minutes=2))

    def generate_otp(self):
        self.otp = str(random.randint(100000, 999999))
        self.otp_created_at = timezone.now()
        self.save()
        return self.otp


    class Meta:
        ordering = ('-date_joined',)