from django.core.validators import RegexValidator
from django.utils.translation import gettext_lazy as _
from django.utils import timezone
from rest_framework import serializers
from apps.core.utils import send_sms
from .models import User

class SignUp_LoginSerializer(serializers.Serializer):
    USER_DETAIL_MODEL = None
    mobile_phone = serializers.CharField(
        max_length=11,
        validators=[
            RegexValidator(
                regex=r'^09\d{9}$',
                message=_("شماره تلفن باید 11 رقمی باشد و با 09 شروع شود")
            )
        ]
    )

    def validate(self, values):
        mobile_phone = values.get('mobile_phone')

        try:
            user = User.objects.get(mobile_phone=mobile_phone)
        except User.DoesNotExist:
            user = User.objects.create(mobile_phone=mobile_phone, username=mobile_phone)

        user_detail, created = self.USER_DETAIL_MODEL.objects.get_or_create(user=user)
        if created:
            user_detail.initialize()
        
        otp = user.generate_otp()
        send_sms(user.mobile_phone, otp)
        values['user_is_new'] = created
        return values

    def create(self, validated_data):
        return validated_data

class OTPVerificationSerializer(serializers.Serializer):
    USER_TYPE = None # Should set to doctor or patient when inherited
    
    otp = serializers.CharField()
    mobile_phone = serializers.CharField()

    def validate(self, values):
        values = super().validate(values)
        try:
            user = User.objects.get(mobile_phone=values['mobile_phone'], otp=values['otp'])
        except User.DoesNotExist:
            raise serializers.ValidationError("کد یکبار مصرف صحیح نیست.")
        
        if self.USER_TYPE == 'doctor':
            if not user.is_doctor():
                raise serializers.ValidationError("خطا در پردازش کاربر")
        elif self.USER_TYPE == 'patient':
            if not user.is_patient():
                raise serializers.ValidationError("خطا در پردازش کاربر")
        
        if user.is_otp_expired():
            # TODO: check how many times user has requested an otp in the past 5 hours before sending a new otp.
            new_otp = user.generate_otp()
            send_sms(user.mobile_phone, new_otp)
            raise serializers.ValidationError("کد یکبار مصرف منقضی شده است . کد دیگری برای شما ارسال خواهد شد")

        values["user"] = user

        return values

    def create(self, validated_data):
        return validated_data

    def save(self, **kwargs):
        user = self.validated_data['user']
        user.last_login = timezone.now()
        user.save()